// Copyright (C) 2022 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only WITH Qt-GPL-exception-1.0

#pragma once

#include "utils_global.h"

#include "futuresynchronizer.h"
#include "qtcassert.h"
#include "threadutils.h"

#include <QtTaskTree/QTaskTree>

#include <QFutureWatcher>
#include <QtConcurrentRun>

namespace Utils {

QTCREATOR_UTILS_EXPORT QThreadPool *asyncThreadPool(QThread::Priority priority);

template <typename Function, typename ...Args>
auto asyncRun(QThreadPool *threadPool, QThread::Priority priority,
              Function &&function, Args &&...args)
{
    QThreadPool *pool = threadPool ? threadPool : asyncThreadPool(priority);
    return QtConcurrent::run(pool, std::forward<Function>(function), std::forward<Args>(args)...);
}

template <typename Function, typename ...Args>
auto asyncRun(QThread::Priority priority, Function &&function, Args &&...args)
{
    return asyncRun<Function, Args...>(nullptr, priority,
                    std::forward<Function>(function), std::forward<Args>(args)...);
}

template <typename Function, typename ...Args>
auto asyncRun(QThreadPool *threadPool, Function &&function, Args &&...args)
{
    return asyncRun<Function, Args...>(threadPool, QThread::InheritPriority,
                    std::forward<Function>(function), std::forward<Args>(args)...);
}

template <typename Function, typename ...Args>
auto asyncRun(Function &&function, Args &&...args)
{
    return asyncRun<Function, Args...>(nullptr, QThread::InheritPriority,
                                       std::forward<Function>(function), std::forward<Args>(args)...);
}

/*!
    Adds a handler for when a result is ready. The guard object determines the lifetime of
    the connection.
    This creates a new QFutureWatcher. Do not use if you intend to react on multiple conditions
    or create a QFutureWatcher already for other reasons.
*/
template <typename T, typename Function>
const QFuture<T> &onResultReady(const QFuture<T> &future, QObject *guard, Function f)
{
    auto watcher = new QFutureWatcher<T>(guard);
    QObject::connect(watcher, &QFutureWatcherBase::finished, watcher, &QObject::deleteLater);
    QObject::connect(watcher, &QFutureWatcherBase::resultReadyAt, guard, [f, watcher](int index) {
        f(watcher->future().resultAt(index));
    });
    watcher->setFuture(future);
    return future;
}

/*!
    Adds a handler for when the future is finished. The guard object determines the lifetime of
    the connection.
    This creates a new QFutureWatcher. Do not use if you intend to react on multiple conditions
    or create a QFutureWatcher already for other reasons.
*/
template<typename T, typename Function>
const QFuture<T> &onFinished(const QFuture<T> &future, QObject *guard, Function f)
{
    auto watcher = new QFutureWatcher<T>(guard);
    QObject::connect(watcher, &QFutureWatcherBase::finished, watcher, &QObject::deleteLater);
    QObject::connect(watcher, &QFutureWatcherBase::finished, guard, [f, watcher] {
        f(watcher->future());
    });
    watcher->setFuture(future);
    return future;
}

class QTCREATOR_UTILS_EXPORT AsyncBase : public QObject
{
    Q_OBJECT

signals:
    void started();
    void done();
    void resultReadyAt(int index);
    void resultsReadyAt(int beginIndex, int endIndex);
    void progressRangeChanged(int min, int max);
    void progressValueChanged(int value);
    void progressTextChanged(const QString &text);
};

template <typename ResultType>
class Async : public AsyncBase
{
public:
    Async()
        : m_synchronizer(isMainThread() ? futureSynchronizer() : nullptr)
    {
        connect(&m_watcher, &QFutureWatcherBase::finished, this, &AsyncBase::done);
        connect(&m_watcher, &QFutureWatcherBase::resultReadyAt, this, &AsyncBase::resultReadyAt);
        connect(&m_watcher, &QFutureWatcherBase::resultsReadyAt, this, &AsyncBase::resultsReadyAt);
        connect(&m_watcher, &QFutureWatcherBase::progressValueChanged,
                this, &AsyncBase::progressValueChanged);
        connect(&m_watcher, &QFutureWatcherBase::progressRangeChanged,
                this, &AsyncBase::progressRangeChanged);
        connect(&m_watcher, &QFutureWatcherBase::progressTextChanged,
                this, &AsyncBase::progressTextChanged);
    }
    ~Async()
    {
        if (isDone())
            return;

        m_watcher.cancel();
        if (!m_synchronizer)
            m_watcher.waitForFinished();
    }

    template <typename Function, typename ...Args>
    void setConcurrentCallData(Function &&function, Args &&...args)
    {
        wrapConcurrent(std::forward<Function>(function), std::forward<Args>(args)...);
    }

    void setFutureSynchronizer(FutureSynchronizer *synchronizer) { m_synchronizer = synchronizer; }
    void setThreadPool(QThreadPool *pool) { m_threadPool = pool; }
    void setPriority(QThread::Priority priority) { m_priority = priority; }

    void start()
    {
        QTC_ASSERT(m_startHandler, qWarning("No start handler specified."); return);
        m_watcher.setFuture(m_startHandler());
        emit started();
        if (m_synchronizer)
            m_synchronizer->addFuture(m_watcher.future());
    }

    bool isDone() const { return m_watcher.isFinished(); }
    bool isCanceled() const { return m_watcher.isCanceled(); }

    QFuture<ResultType> future() const { return m_watcher.future(); }
    ResultType result() const { return m_watcher.result(); }
    ResultType takeResult() const { return m_watcher.future().takeResult(); }
    ResultType resultAt(int index) const { return m_watcher.resultAt(index); }
    QList<ResultType> results() const { return future().results(); }
    bool isResultAvailable() const { return future().resultCount(); }

private:
    template <typename Function, typename ...Args>
    void wrapConcurrent(Function &&function, Args &&...args)
    {
        m_startHandler = [this, function = std::forward<Function>(function), args...] {
            return asyncRun(m_threadPool, m_priority, function, args...);
        };
    }

    template <typename Function, typename ...Args>
    void wrapConcurrent(std::reference_wrapper<const Function> &&wrapper, Args &&...args)
    {
        m_startHandler = [this, wrapper = std::forward<std::reference_wrapper<const Function>>(wrapper), args...] {
            return asyncRun(m_threadPool, m_priority, std::forward<const Function>(wrapper.get()),
                            args...);
        };
    }

    using StartHandler = std::function<QFuture<ResultType>()>;
    StartHandler m_startHandler;
    FutureSynchronizer *m_synchronizer = nullptr;
    QThreadPool *m_threadPool = nullptr;
    QThread::Priority m_priority = QThread::InheritPriority;
    QFutureWatcher<ResultType> m_watcher;
};

template <typename ResultType>
class AsyncTaskAdapter final
{
public:
    void operator()(Async<ResultType> *task, QTaskInterface *iface) {
        QObject::connect(task, &AsyncBase::done, iface, [iface, task] {
            iface->reportDone(QtTaskTree::toDoneResult(!task->isCanceled()));
        }, Qt::SingleShotConnection);
        task->start();
    }
};

template <typename T>
using AsyncTask = QCustomTask<Async<T>, AsyncTaskAdapter<T>>;

} // namespace Utils
