/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGDOCUMENTBANK_H
#define SKGDOCUMENTBANK_H
/** @file
 * This file defines classes SKGDocumentBank.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "skgdocument.h"
#include "skgdefinebank.h"
#include "skgbankmodeler_export.h"

class SKGUnitValueObject;

/**
 * This class manages skg bank documents
 */
class SKGBANKMODELER_EXPORT SKGDocumentBank : public SKGDocument
{
    Q_OBJECT
    Q_CLASSINFO("D-Bus Interface", "org.kde.skrooge.SKGDocumentBank")

public:
    /**
    * Constructor
    */
    explicit SKGDocumentBank();

    /**
    * Destructor
    */
    virtual ~SKGDocumentBank();

public Q_SLOTS:
    /**
     * Close the current transaction.
     * A transaction is needed to modify the SKGDocument.
     * This transaction is also used to manage the undo/redo.
     * @see beginTransaction
     * @param succeedded : true to indicate that current transaction has been successfully executed
     *                   : false to indicate that current transaction has failed
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError endTransaction(bool succeedded) override;

    /**
     * dump the document in the std output.
     * It is useful for debug.
     * @param iMode to select what you want to dump.
     * @code
     * document->dump (DUMPUNIT|DUMPPARAMETERS);
     * @endcode
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError dump(int iMode = DUMPBANKOBJECT) override;

    /**
     * Create or modify an account
     * @param iName account name
     * @param iNumber account number
     * @param iBankName name of the bank
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError addOrModifyAccount(const QString& iName, const QString& iNumber, const QString& iBankName);

    /**
     * Create or modify the value of an unit
     * @param iUnitName unit name
     * @param iDate date
     * @param iValue unit value for the date @p iDate
     * @param oValue this output unit value
     * @return an object managing the error.
     *   @see SKGError
     */
    // cppcheck-suppress passedByValue
    virtual SKGError addOrModifyUnitValue(const QString& iUnitName, const QDate iDate, double iValue, SKGUnitValueObject* oValue = nullptr);

    /**
     * Get Primary unit. WARNING: This value can be not uptodated in a transaction.
     * @return Primary unit.
     */
    virtual SKGServices::SKGUnitInfo getPrimaryUnit();

    /**
     * Get Secondary unit. WARNING: This value can be not uptodated in a transaction.
     * @return Secondary unit.
     */
    virtual SKGServices::SKGUnitInfo getSecondaryUnit();

    /**
     * Get formated money in primary unit
     * @param iValue value
     * @return formated value in red or black
     */
    virtual QString formatPrimaryMoney(double iValue) override;

    /**
     * Get formated money in primary unit
     * @param iValue value
     * @return formated value in red or black
     */
    virtual QString formatSecondaryMoney(double iValue) override;

    /**
    * Get the preferred category for a payee. WARNING: This value can be not uptodated in a transaction.
    * @param iPayee the payee
    * @return The preferred category.
    */
    virtual QString getCategoryForPayee(const QString& iPayee);

    /**
     * Refresh all views and indexes in the database
     * @param iForce force the refresh
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError refreshViewsIndexesAndTriggers(bool iForce = false) override;

    /**
     * Return the number version of views, indexes and triggers
     * @return
     */
    virtual QString getViewsIndexesAndTriggersVersion();

    /**
     * Get display schemas
     * @param iRealTable the real table name
     * @return list of schemas
     */
    virtual SKGDocument::SKGModelTemplateList getDisplaySchemas(const QString& iRealTable) const override;

    /**
     * Get the display string for any modeler object (table, attribute)
     * @param iString the name of the object (example: v_operation, v_unit.t_name)
     * @return the display string
     */
    virtual QString getDisplay(const QString& iString) const override;

    /**
     * Get the real attribute
     * @param iString the name of the attribute (something like t_BANK)
     * @return the real attribute (something like bank.rd_bank_id.t_name)
     */
    virtual QString getRealAttribute(const QString& iString) const override;

    /**
     * Get the icon for attribute
     * @param iString the name of the attribute
     * @return the icon
     */
    virtual QIcon getIcon(const QString& iString) const override;

    /**
     * Get the attribute type
     * @param iAttributeName the name of an attribute
     * @return the type
     */
    virtual SKGServices::AttributeType getAttributeType(const QString& iAttributeName) const override;

    /**
     * Get the file extension for this kind of document (must be overwritten)
     * @return file extension (like skg)
     */
    virtual QString getFileExtension() const override;

    /**
     * Get the header of the file (useful for magic mime type).
     * @return document header
     */
    virtual QString getDocumentHeader() const override;

    /**
     * Get budget report
     * @param iMonth the month
     * @return the report
     */
    virtual QVariantList getBudget(const QString& iMonth);

    /**
     * Get main categorys report
     * @param iPeriod the period
     * @param iNb number of categories
     * @return the report
     */
    virtual QVariantList getMainCategories(const QString& iPeriod, int iNb = 5);

    /**
     * Get 5 main variation of categories report
     * @param iPeriod the period
     * @param iPreviousPeriod the previous period
     * @param iOnlyIssues only "Expenses increased" and "Incomes decreased"
     * @param oCategoryList to get the category for each variation
     * @return the list of variation string
     */
    virtual QStringList get5MainCategoriesVariationList(const QString& iPeriod, const QString& iPreviousPeriod, bool iOnlyIssues, QStringList* oCategoryList = nullptr);

    /**
     * Enable/disable balances computation
     */
    virtual void setComputeBalances(bool iEnabled);

    /**
     * Get the report
     * Do not forget to delete the pointer
     * @return the report
     */
    virtual SKGReport* getReport() override;

    /**
     * Refresh the case.
     * @param iTable the modified table triggering the cache refresh.
     */
    virtual void refreshCache(const QString& iTable) override;

protected:
    /**
     * Migrate the current SKGDocument to the latest version of the data model.
     * WARNING: This method must be used in a transaction.
     * @see beginTransaction
     * @param oMigrationDone to know if a migration has been done or not.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError migrate(bool& oMigrationDone) override;

    /**
     * Compute balance of each operation.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError computeBalances();

    /**
     * Compute the budget suboperation links.
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError computeBudgetSuboperationLinks();

private:
    Q_DISABLE_COPY(SKGDocumentBank)

    QString m_5mainVariations_inputs;
    QStringList m_5mainVariations_cache;
    QStringList m_5mainVariationsCat_cache;
    bool m_computeBalances;
};
/**
 * Declare the class
 */
Q_DECLARE_TYPEINFO(SKGDocumentBank, Q_MOVABLE_TYPE);
#endif
