/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtestmacro.h"
#include "skgbankincludes.h"
#include "skgimportexportmanager.h"

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)

    QDate d(1970, 1, 1);

    {
        // 275963
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGBankObject bank(&document1);
        SKGUnitObject unit_euro(&document1);
        SKGUnitObject unit_dollar(&document1);
        SKGError err;
        {
            SKGAccountObject account;

            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_INIT"), err)

            // Creation unit
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit_euro.setName(QLatin1String("euro")), true)
            SKGTESTERROR(QLatin1String("UNIT:setSymbol"), unit_euro.setSymbol(QLatin1String("EUR")), true)
            SKGTESTERROR(QLatin1String("UNIT:setType"), unit_euro.setType(SKGUnitObject::PRIMARY), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit_euro.save(), true)

            // Creation unitvalue
            SKGUnitValueObject unit_euro_val1;
            SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit_euro.addUnitValue(unit_euro_val1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setQuantity"), unit_euro_val1.setQuantity(1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setDate"), unit_euro_val1.setDate(d), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:save"), unit_euro_val1.save(), true)

            // Creation unit
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit_dollar.setName(QLatin1String("dollar")), true)
            SKGTESTERROR(QLatin1String("UNIT:setSymbol"), unit_dollar.setSymbol(QLatin1String("USD")), true)
            SKGTESTERROR(QLatin1String("UNIT:setType"), unit_dollar.setType(SKGUnitObject::CURRENCY), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit_dollar.save(), true)

            // Creation unitvalue
            SKGUnitValueObject unit_dollar_val1;
            SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit_dollar.addUnitValue(unit_dollar_val1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setQuantity"), unit_dollar_val1.setQuantity(2), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setDate"), unit_dollar_val1.setDate(d), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:save"), unit_dollar_val1.save(), true)

            // Creation bank
            SKGTESTERROR(QLatin1String("BANK:setName"), bank.setName(QLatin1String("CREDIT COOP")), true)
            SKGTESTERROR(QLatin1String("BANK:setNumber"), bank.setNumber(QLatin1String("0003")), true)
            SKGTESTERROR(QLatin1String("BANK:save"), bank.save(), true)

            // Creation account
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank.addAccount(account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account.setName(QLatin1String("CODEVI")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account.save(), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setInitialBalance"), account.setInitialBalance(100, unit_dollar), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account.save(), true)
        }


        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/t2.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("CODEVI")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-9335.94"))
        }
    }

    {
        // 275963
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.load(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/export_qif.skg"), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_BP_QIF"), err)
            SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif2/export_qif.qif"));
            SKGTESTERROR(QLatin1String("QIF.exportFile"), exp1.exportFile(), true)
        }
    }

    {
        // Support qif file having Type:Class with description
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        SKGAccountObject la;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/MoneydanceExportExample.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)

            SKGObjectBase::SKGListSKGObjectBase result;
            SKGTESTERROR(QLatin1String("DOC.getObjects"), document1.getObjects(QLatin1String("account"), QLatin1String("t_name='Wells Fargo:Checking'"), result), true)
            SKGTEST(QLatin1String("DOC.getObjects.count"), result.count(), 1)
            if (result.count() != 0) {
                la = result.at(0);
            }
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_QIF"), err)
            SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif2/export_all.qif"));
            SKGTESTERROR(QLatin1String("QIF.exportFile"), exp1.exportFile(), true)
        }
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("EXPORT_QIF"), err)
            SKGImportExportManager exp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestimportqif2/export_la.qif"));
            QMap<QString, QString> params;
            params[QLatin1String("uuid_of_selected_accounts_or_operations")] = la.getUniqueID();
            exp1.setExportParameters(params);
            SKGTESTERROR(QLatin1String("QIF.exportFile"), exp1.exportFile(), true)
        }
    }

    {
        // Default account
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/LA-1234567@ing.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGBEGINTRANSACTION(document1, QLatin1String("MODIF"), err)
            SKGObjectBase::SKGListSKGObjectBase result;
            SKGTESTERROR(QLatin1String("DOC.getObjects"), document1.getObjects(QLatin1String("bank"), QLatin1String("t_name='ing'"), result), true)
            SKGTEST(QLatin1String("DOC.getObjects.count"), result.count(), 1)

            SKGBankObject bank(result.at(0));
            SKGTESTERROR(QLatin1String("BANK.setName"), bank.setName(QLatin1String("ING Direct")), true)
            SKGTESTERROR(QLatin1String("BANK.save"), bank.save(), true)

            SKGTESTERROR(QLatin1String("DOC.getObjects"), document1.getObjects(QLatin1String("account"), QLatin1String("t_number='1234567'"), result), true)
            SKGTEST(QLatin1String("DOC.getObjects.count"), result.count(), 1)

            SKGTESTERROR(QLatin1String("DOC.getObjects"), document1.getObjects(QLatin1String("account"), QLatin1String("t_name='LA'"), result), true)
            SKGTEST(QLatin1String("DOC.getObjects.count"), result.count(), 1)

            SKGAccountObject acc(result.at(0));
            SKGTESTERROR(QLatin1String("ACC.setName"), acc.setName(QLatin1String("Livret A")), true)
            SKGTESTERROR(QLatin1String("ACC.save"), acc.save(), true)
        }

        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/LA-1234567@ing.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        document1.dump(DUMPACCOUNT);
        {
            SKGObjectBase::SKGListSKGObjectBase result;
            SKGTESTERROR(QLatin1String("DOC.getObjects"), document1.getObjects(QLatin1String("account"), QLatin1String("t_number='1234567'"), result), true)
            SKGTEST(QLatin1String("DOC.getObjects.count"), result.count(), 1)
        }
    }

    {
        // Bug import date
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/date_money.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
    }

    {
        // Autorepair
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/autorepair.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("autorepair")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-400"))
        }
    }

    {
        // Error
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/error.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
    }

    {
        // Split and transfer
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/split_and_transfer.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("Joint Checking")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("-1181.25"))  // -600 if sum of splits
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("2012 Acadia Loan")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("1162.5"))  // 581.25 if sum of splits
        }
    }

    {
        // Split and transfer 2
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/split_and_transfer_2.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("Joint Checking")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("2714.31"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("Credit Union")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("400"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("TSP")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("437.88"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("FSA")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("88.26"))
        }
        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("TSP - Roth")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("250"))
        }
    }

    {
        // 400724
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/400724.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
    }

    {
        // 402330
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/402330.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
    }

    {
        // 403725
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/403725.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        SKGObjectBase::SKGListSKGObjectBase result;
        SKGTESTERROR(QLatin1String("DOC.getObjects"), document1.getObjects(QLatin1String("refund"), QLatin1String(""), result), true)
        SKGTEST(QLatin1String("DOC.getObjects.count"), result.count(), 2)
    }

    {
        // 406243
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/406243.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGAccountObject account(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), account.setName(QLatin1String("406243")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), account.load(), true)
            SKGTEST(QLatin1String("QIF:getValue"), SKGServices::doubleToString(account.getCurrentAmount()), QLatin1String("794"))
        }
    }

    {
        // 406243
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/406271.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGTrackerObject tracker(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), tracker.setName(QLatin1String("Jane")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), tracker.load(), true)
            SKGTEST(QLatin1String("QIF:getComment"), tracker.getComment(), QLatin1String("expenses that Jane incurs"))
        }
    }

    {
        // 406266
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/406266.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        {
            SKGUnitObject unit(&document1);
            SKGTESTERROR(QLatin1String("QIF.setName"), unit.setName(QLatin1String("ACME CORP")), true)
            SKGTESTERROR(QLatin1String("QIF.load"), unit.load(), true)
            SKGTEST(QLatin1String("QIF:getSymbol"), unit.getSymbol(), QLatin1String("ACMW"))
            SKGTEST(QLatin1String("QIF:getSymbol"), static_cast<unsigned int>(unit.getType()), static_cast<unsigned int>(SKGUnitObject::SHARE))
        }
    }

    {
        // 406270
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/406270.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        SKGObjectBase::SKGListSKGObjectBase result;
        SKGTESTERROR(QLatin1String("DOC.getObjects"), document1.getObjects(QLatin1String("unit"), QLatin1String(""), result), true)
        SKGTEST(QLatin1String("DOC.getObjects.count"), static_cast<unsigned int>(result.count()), static_cast<unsigned int>(1))
    }

    {
        // BUG
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/bug.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }
    }

    {
        // 489784
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGBankObject bank(&document1);
        SKGAccountObject account;
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT_INIT"), err)

            // Creation bank
            SKGTESTERROR(QLatin1String("BANK:setName"), bank.setName(QLatin1String("CREDIT COOP")), true)
            SKGTESTERROR(QLatin1String("BANK:setNumber"), bank.setNumber(QLatin1String("0003")), true)
            SKGTESTERROR(QLatin1String("BANK:save"), bank.save(), true)

            // Creation account
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank.addAccount(account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account.setName(QLatin1String("489784")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setType"), account.setType(SKGAccountObject::CREDITCARD), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account.save(), true)
        }

        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/489784.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        SKGTESTERROR(QLatin1String("ACCOUNT:load"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getType"), static_cast<unsigned int>(account.getType()), static_cast<unsigned int>(SKGAccountObject::CREDITCARD))
    }

    {
        // 493062
        SKGDocumentBank document1;
        SKGError err;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true) {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("IMPORT"), err)

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath(QLatin1String("IN")) % "/skgtestimportqif2/493062.qif"));
            SKGTESTERROR(QLatin1String("QIF.importFile"), imp1.importFile(), true)
        }

        SKGObjectBase::SKGListSKGObjectBase result;
        SKGTESTERROR(QLatin1String("DOC.getObjects"), document1.getObjects(QLatin1String("refund"), QLatin1String("1=1"), result), true)
        SKGTEST(QLatin1String("DOC.getObjects.count"), result.count(), 0)
    }

    // End test
    SKGENDTEST()
}
