/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtestmacro.h"
#include "skgbankincludes.h"

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)
    // ============================================================================
    // Init
    {
        // Test bank1 document
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("BANK_T1"), err)

            // Creation unit
            SKGUnitObject unit_euro(&document1);
            SKGTESTERROR(QLatin1String("BANK:setName"), unit_euro.setName(QLatin1String("euro")), true)
            SKGTESTERROR(QLatin1String("BANK:setSymbol"), unit_euro.setSymbol(QLatin1String("euro")), true)
            SKGTESTERROR(QLatin1String("BANK:save"), unit_euro.save(), true)

            // Creation unitvalue
            SKGUnitValueObject unit_euro_val1;
            SKGTESTERROR(QLatin1String("BANK:addUnitValue"), unit_euro.addUnitValue(unit_euro_val1), true)
            SKGTESTERROR(QLatin1String("BANK:setQuantity"), unit_euro_val1.setQuantity(1), true)
            SKGTESTERROR(QLatin1String("BANK:setDate"), unit_euro_val1.setDate(QDate::currentDate()), true)
            SKGTESTERROR(QLatin1String("BANK:save"), unit_euro_val1.save(), true)

            // Creation bank1
            SKGBankObject bank1(&document1);
            SKGAccountObject account1;
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank1.addAccount(account1), false)

            SKGTESTERROR(QLatin1String("BANK:setName"), bank1.setName(QLatin1String("CREDIT COOP")), true)
            SKGTESTERROR(QLatin1String("BANK:setIcon"), bank1.setIcon(QLatin1String("credit cooperatif")), true)
            SKGTESTERROR(QLatin1String("BANK:setNumber"), bank1.setNumber(QLatin1String("111111")), true)
            SKGTESTERROR(QLatin1String("BANK:save"), bank1.save(), true)
            SKGTEST(QLatin1String("BANK:getIcon"), bank1.getIcon(), QLatin1String("credit cooperatif"))
            SKGTEST(QLatin1String("BANK:getNumber"), bank1.getNumber(), QLatin1String("111111"))

            SKGBankObject bank11(bank1);    // For coverage
            SKGTESTBOOL("BANK:comparison", (bank11 == bank1), true)
            SKGBankObject bank12 = bank1;  // For coverage
            SKGTESTBOOL("BANK:comparison", (bank12 == bank1), true)
            SKGBankObject bank13((SKGNamedObject(bank1)));  // For coverage
            SKGTESTBOOL("BANK:comparison", (bank12 == SKGNamedObject(bank1)), true)
            SKGNamedObject bank1n(bank1);
            SKGBankObject bank14(bank1n);  // For coverage

            SKGBankObject bank15(SKGObjectBase(bank1.getDocument(), QLatin1String("xxx"), bank1.getID()));    // For coverage
            SKGBankObject bank16(SKGNamedObject(bank1.getDocument(), QLatin1String("xxx"), bank1.getID()));    // For coverage

            // Creation account1
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank1.addAccount(account1), true)

            SKGOperationObject op;
            SKGTESTERROR(QLatin1String("ACCOUNT:addOperation"), account1.addOperation(op), false)

            SKGInterestObject it;
            SKGTESTERROR(QLatin1String("ACCOUNT:addOperation"), account1.addInterest(it), false)

            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account1.setName(QLatin1String("Courant steph")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setNumber"), account1.setNumber(QLatin1String("12345P")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setAgencyNumber"), account1.setAgencyNumber(QLatin1String("99999")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setAgencyAddress"), account1.setAgencyAddress(QLatin1String("10 rue Albert CAMUS\n31000 TOULOUSE")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setType"), account1.setType(SKGAccountObject::CURRENT), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setComment"), account1.setComment(QLatin1String("bla bla")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setMaxLimitAmount"), account1.setMaxLimitAmount(15000.0), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setMinLimitAmount"), account1.setMinLimitAmount(-500.0), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:maxLimitAmountEnabled"), account1.maxLimitAmountEnabled(true), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:minLimitAmountEnabled"), account1.minLimitAmountEnabled(false), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account1.save(), true)
            SKGTEST(QLatin1String("ACCOUNT:getNumber"), account1.getNumber(), QLatin1String("12345P"))
            SKGTEST(QLatin1String("ACCOUNT:getAgencyNumber"), account1.getAgencyNumber(), QLatin1String("99999"))
            SKGTEST(QLatin1String("ACCOUNT:getAgencyAddress"), account1.getAgencyAddress(), QLatin1String("10 rue Albert CAMUS\n31000 TOULOUSE"))
            SKGTEST(QLatin1String("ACCOUNT:getMaxLimitAmount"), account1.getMaxLimitAmount(), 15000.0)
            SKGTEST(QLatin1String("ACCOUNT:getMinLimitAmount"), account1.getMinLimitAmount(), -500.0)

            SKGTESTERROR(QLatin1String("ACCOUNT:setMaxLimitAmount"), account1.setMaxLimitAmount(-1000.0), true)
            SKGTEST(QLatin1String("ACCOUNT:getMaxLimitAmount"), account1.getMaxLimitAmount(), -1000.0)
            SKGTEST(QLatin1String("ACCOUNT:getMinLimitAmount"), account1.getMinLimitAmount(), -1000.0)

            SKGTESTBOOL("ACCOUNT:isMaxLimitAmountEnabled", account1.isMaxLimitAmountEnabled(), true)
            SKGTESTBOOL("ACCOUNT:isMinLimitAmountEnabled", account1.isMinLimitAmountEnabled(), false)

            SKGTEST(QLatin1String("ACCOUNT:getType"), static_cast<unsigned int>(account1.getType()), static_cast<unsigned int>(SKGAccountObject::CURRENT))

            SKGTESTERROR(QLatin1String("ACCOUNT:setType"), account1.setType(SKGAccountObject::CREDITCARD), true)
            SKGTEST(QLatin1String("ACCOUNT:getType"), static_cast<unsigned int>(account1.getType()), static_cast<unsigned int>(SKGAccountObject::CREDITCARD))

            SKGTESTERROR(QLatin1String("ACCOUNT:setType"), account1.setType(SKGAccountObject::ASSETS), true)
            SKGTEST(QLatin1String("ACCOUNT:getType"), static_cast<unsigned int>(account1.getType()), static_cast<unsigned int>(SKGAccountObject::ASSETS))

            SKGTESTERROR(QLatin1String("ACCOUNT:setType"), account1.setType(SKGAccountObject::INVESTMENT), true)
            SKGTEST(QLatin1String("ACCOUNT:getType"), static_cast<unsigned int>(account1.getType()), static_cast<unsigned int>(SKGAccountObject::INVESTMENT))

            SKGTESTERROR(QLatin1String("ACCOUNT:setType"), account1.setType(SKGAccountObject::WALLET), true)
            SKGTEST(QLatin1String("ACCOUNT:getType"), static_cast<unsigned int>(account1.getType()), static_cast<unsigned int>(SKGAccountObject::WALLET))

            SKGTESTERROR(QLatin1String("ACCOUNT:setType"), account1.setType(SKGAccountObject::LOAN), true)
            SKGTEST(QLatin1String("ACCOUNT:getType"), static_cast<unsigned int>(account1.getType()), static_cast<unsigned int>(SKGAccountObject::LOAN))

            SKGTESTERROR(QLatin1String("ACCOUNT:setType"), account1.setType(SKGAccountObject::SAVING), true)
            SKGTEST(QLatin1String("ACCOUNT:getType"), static_cast<unsigned int>(account1.getType()), static_cast<unsigned int>(SKGAccountObject::SAVING))

            SKGTESTERROR(QLatin1String("ACCOUNT:setType"), account1.setType(SKGAccountObject::OTHER), true)
            SKGTEST(QLatin1String("ACCOUNT:getType"), static_cast<unsigned int>(account1.getType()), static_cast<unsigned int>(SKGAccountObject::OTHER))

            SKGTESTERROR(QLatin1String("ACCOUNT:setType"), account1.setType(SKGAccountObject::PENSION), true)
            SKGTEST(QLatin1String("ACCOUNT:getType"), static_cast<unsigned int>(account1.getType()), static_cast<unsigned int>(SKGAccountObject::PENSION))

            SKGTEST(QLatin1String("ACCOUNT:getComment"), account1.getComment(), QLatin1String("bla bla"))

            QDate n = QDate::currentDate();
            SKGTESTERROR(QLatin1String("ACCOUNT:setReconciliationDate"), account1.setReconciliationDate(n), true)
            SKGTEST(QLatin1String("ACCOUNT:getReconciliationDate"), SKGServices::dateToSqlString(account1.getReconciliationDate()), SKGServices::dateToSqlString(n))
            SKGTESTERROR(QLatin1String("ACCOUNT:setReconciliationBalance"), account1.setReconciliationBalance(125), true)
            SKGTEST(QLatin1String("ACCOUNT:getReconciliationBalance"), SKGServices::doubleToString(account1.getReconciliationBalance()), SKGServices::doubleToString(125))

            SKGTESTERROR(QLatin1String("ACCOUNT:setInitialBalance"), account1.setInitialBalance(125, unit_euro), true)
            double init = 0;
            SKGUnitObject unit;
            SKGTESTERROR(QLatin1String("ACCOUNT:getInitialBalance"), account1.getInitialBalance(init, unit), true)
            SKGTESTBOOL("ACCOUNT:getInitialBalance", (unit == unit_euro), true)
            SKGTEST(QLatin1String("ACCOUNT:getInitialBalance"), init, 125)

            SKGAccountObject account11(account1);    // For coverage
            SKGTESTBOOL("BANK:comparison", (account11 == account1), true)
            SKGAccountObject account12(static_cast<SKGNamedObject>(account1));     // For coverage
            SKGTESTBOOL("BANK:comparison", (account12 == account1), true)
            SKGAccountObject account13(static_cast<SKGObjectBase>(account1));     // For coverage
            SKGTESTBOOL("BANK:comparison", (account13 == account1), true)

            SKGAccountObject account14(SKGObjectBase(account1.getDocument(), QLatin1String("xxx"), account1.getID()));    // For coverage
            SKGAccountObject account15(SKGNamedObject(account1.getDocument(), QLatin1String("xxx"), account1.getID()));    // For coverage

            // Creation bank2
            SKGBankObject bank2(&document1);
            SKGTESTERROR(QLatin1String("BANK:setName"), bank2.setName(QLatin1String("NEF")), true)
            SKGTESTERROR(QLatin1String("BANK:save"), bank2.save(), true)

            // Creation account2
            SKGAccountObject account2;
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank2.addAccount(account2), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account2.setName(QLatin1String("Courant steph")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setNumber"), account2.setNumber(QLatin1String("98765A")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account2.save(false), false)

            SKGAccountObject account3;
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank2.addAccount(account3), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account3.setName(QLatin1String("Courant vero")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setNumber"), account3.setNumber(QLatin1String("98765A")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account3.save(), true)

            QStringList oResult;
            SKGTESTERROR(QLatin1String("BANK:getDistinctValues"), document1.getDistinctValues(QLatin1String("bank"), QLatin1String("t_name"), oResult), true)
            SKGTEST(QLatin1String("BANK:oResult.size"), oResult.size(), 2)

            SKGTESTERROR(QLatin1String("BANK:getDistinctValues"), document1.getDistinctValues(QLatin1String("account"), QLatin1String("t_name"), oResult), true)
            SKGTEST(QLatin1String("BANK:oResult.size"), oResult.size(), 2)

            SKGObjectBase::SKGListSKGObjectBase oAccountList;
            SKGTESTERROR(QLatin1String("BANK:getAccounts"), bank1.getAccounts(oAccountList), true)
            SKGTEST(QLatin1String("ACCOUNT:count"), oAccountList.count(), 1)

            SKGTESTERROR(QLatin1String("BANK:getAccounts"), bank2.getAccounts(oAccountList), true)
            SKGTEST(QLatin1String("ACCOUNT:count"), oAccountList.count(), 1)

            // Modification account1
            SKGBankObject tmpBank;
            SKGTESTERROR(QLatin1String("ACCOUNT:getBank"), account3.getBank(tmpBank), true)
            SKGTESTBOOL("BANK:tmpBank==bank2", (tmpBank == bank2), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setBank"), account3.setBank(bank1), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:getBank"), account3.getBank(tmpBank), true)
            SKGTESTBOOL("BANK:tmpBank==bank2", (tmpBank == bank1), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account3.save(), true)

            SKGTESTERROR(QLatin1String("BANK:getAccounts"), bank1.getAccounts(oAccountList), true)
            SKGTEST(QLatin1String("ACCOUNT:count"), oAccountList.count(), 2)

            SKGTESTERROR(QLatin1String("BANK:getAccounts"), bank2.getAccounts(oAccountList), true)
            SKGTEST(QLatin1String("ACCOUNT:count"), oAccountList.count(), 0)

            SKGTESTBOOL("ACCOUNT:isBookmarked", account3.isBookmarked(), false)
            SKGTESTERROR(QLatin1String("ACCOUNT:bookmark"), account3.bookmark(true), true)
            SKGTESTBOOL("ACCOUNT:isBookmarked", account3.isBookmarked(), true)

            SKGTESTBOOL("ACCOUNT:isClosed", account3.isClosed(), false)
            SKGTESTERROR(QLatin1String("ACCOUNT:setClosed"), account3.setClosed(true), true)
            SKGTESTBOOL("ACCOUNT:isClosed", account3.isClosed(), true)

            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account3.save(), true)

            // Merge
            SKGTESTERROR(QLatin1String("ACCOUNT:merge"), account2.merge(account3, true), true)

            // interest
            SKGObjectBase::SKGListSKGObjectBase oInterestList;
            SKGTESTERROR(QLatin1String("ACCOUNT:getInterests"), account2.getInterests(oInterestList), true)

            SKGInterestObject oInterest;
            SKGTESTERROR(QLatin1String("ACCOUNT:getInterests"), account2.getInterest(QDate::currentDate(), oInterest), false)

            SKGObjectBase account4(&document1, QLatin1String("account"), account1.getID());
            SKGTESTERROR(QLatin1String("ACCOUNT:load"), account4.load(), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setAttribute"), account4.setAttribute(QLatin1String("t_BANK"), QLatin1String("bankname")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account4.save(), true)
            SKGTESTERROR(QLatin1String("BANK:load"), bank1.load(), true)
            SKGTEST(QLatin1String("BANK:getName"), bank1.getName(), QLatin1String("bankname"))

            SKGTESTERROR(QLatin1String("DOC:dump"), document1.dump(DUMPACCOUNT), true)
        }
    }

    // End test
    SKGENDTEST()
}
