/***************************************************************************
 * SPDX-FileCopyrightText: 2024 S. MANKOWSKI stephane@mankowski.fr
 * SPDX-FileCopyrightText: 2024 G. DE BURE support@mankowski.fr
 * SPDX-License-Identifier: GPL-3.0-or-later
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include <qfile.h>

#include "skgbankincludes.h"
#include "skgtestmacro.h"
#include "skgtraces.h"

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc)
    Q_UNUSED(argv)

    // Init test
    SKGINITTEST(true)

    QDate now = QDate::currentDate();

    // ============================================================================
    // Init
    QString filename = SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestbigdocument/skgtestbigdocument.skg";

    {
        // Test bank document
        SKGDocumentBank document1;
        SKGTESTERROR(QLatin1String("document1.initialize()"), document1.initialize(), true)
        SKGUnitValueObject unit_euro_val1;
        SKGBankObject bank(&document1);
        SKGAccountObject account;
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, QLatin1String("BANK_T1"), err)

            // Creation bank
            SKGTESTERROR(QLatin1String("BANK:setName"), bank.setName(QLatin1String("CREDIT COOP")), true)
            SKGTESTERROR(QLatin1String("BANK:save"), bank.save(), true)

            // Creation account
            SKGTESTERROR(QLatin1String("BANK:addAccount"), bank.addAccount(account), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setName"), account.setName(QLatin1String("Courant steph")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:setNumber"), account.setNumber(QLatin1String("12345P")), true)
            SKGTESTERROR(QLatin1String("ACCOUNT:save"), account.save(), true)

            // Creation unit
            SKGUnitObject unit_euro(&document1);
            SKGTESTERROR(QLatin1String("UNIT:setName"), unit_euro.setName(QLatin1String("euro")), true)
            SKGTESTERROR(QLatin1String("UNIT:save"), unit_euro.save(), true)

            // Creation unitvalue
            SKGTESTERROR(QLatin1String("UNIT:addUnitValue"), unit_euro.addUnitValue(unit_euro_val1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setQuantity"), unit_euro_val1.setQuantity(1), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:setDate"), unit_euro_val1.setDate(now), true)
            SKGTESTERROR(QLatin1String("UNITVALUE:save"), unit_euro_val1.save(), true)

            // Creation categories
            auto cats = new SKGCategoryObject[30];
            for (int i = 0; i < 10; ++i) {
                cats[i] = SKGCategoryObject(&document1);
                SKGTESTERROR(QLatin1String("CAT:setName"), cats[i].setName("cat_" % SKGServices::intToString(i)), true)
                SKGTESTERROR(QLatin1String("CAT:save"), cats[i].save(), true)

                for (int j = 0; j < 2; ++j) {
                    int indexSubCat = 10 * (j + 1) + i;
                    SKGTESTERROR(QLatin1String("CAT:addCategory"), cats[i].addCategory(cats[indexSubCat]), true)
                    SKGTESTERROR(QLatin1String("CAT:setName"), cats[indexSubCat].setName("cat_" % SKGServices::intToString(i) % QLatin1Char('_') % SKGServices::intToString(j)), true)
                    SKGTESTERROR(QLatin1String("CAT:save"), cats[indexSubCat].save(), true)
                }
            }

            // Creation payees
            auto payees = new SKGPayeeObject[10];
            for (int i = 0; i < 10; ++i) {
                payees[i] = SKGPayeeObject(&document1);
                SKGTESTERROR(QLatin1String("PAY:setName"), payees[i].setName("pay_" % SKGServices::intToString(i)), true)
                SKGTESTERROR(QLatin1String("PAY:save"), payees[i].save(), true)
            }

            // Mode
            auto modes = new QString[5];
            modes[0] = QLatin1String("cheque");
            modes[1] = QLatin1String("carte");
            modes[2] = QLatin1String("tip");
            modes[3] = QLatin1String("virement");
            modes[4] = QLatin1String("espece");

            // Comments
            auto comments = new QString[3];
            comments[0] = QLatin1String("bla bla");
            comments[1] = QLatin1String("hello world");
            comments[2] = QLatin1String("youpi");

            // Creation operation
            SKGOperationObject mainOperation;
            for (int i = 1; i <= 365 * 10; ++i) {
                SKGOperationObject op_1;
                SKGTESTERROR(QLatin1String("ACCOUNT:addOperation"), account.addOperation(op_1), true)
                SKGTESTERROR(QLatin1String("OPE:setNumber"), op_1.setNumber(SKGServices::intToString(1000 + i)), true)
                SKGTESTERROR(QLatin1String("OPE:setMode"), op_1.setMode(modes[i % 5]), true)
                SKGTESTERROR(QLatin1String("OPE:setComment"), op_1.setComment(comments[i % 3]), true)
                SKGTESTERROR(QLatin1String("OPE:setDate"), op_1.setDate(now.addDays(-i)), true)
                SKGTESTERROR(QLatin1String("OPE:setUnit"), op_1.setUnit(unit_euro), true)
                SKGTESTERROR(QLatin1String("OPE:setStatus"), op_1.setStatus((i < 20 ? SKGOperationObject::NONE : (i < 40 ? SKGOperationObject::MARKED : SKGOperationObject::CHECKED))), true)
                SKGTESTERROR(QLatin1String("OPE:bookmark"), op_1.bookmark(i % 2 == 0), true)
                SKGTESTERROR(QLatin1String("OPE:save"), op_1.save(), true)
                if (i == 1) {
                    mainOperation = op_1;
                    mainOperation.setGroupOperation(mainOperation);
                    SKGTESTERROR(QLatin1String("OPE:save"), mainOperation.save(), true)
                } else {
                    if (!op_1.isBookmarked()) {
                        op_1.setGroupOperation(mainOperation);
                    }
                    SKGTESTERROR(QLatin1String("OPE:save"), op_1.save(), true)
                }

                // Creation suboperation
                for (int j = 1; j <= 2; ++j) {
                    SKGSubOperationObject subop_1;
                    SKGTESTERROR(QLatin1String("OPE:addSubOperation"), op_1.addSubOperation(subop_1), true)
                    SKGTESTERROR(QLatin1String("SUBOPE:setCategory"), subop_1.setCategory(cats[i % 30]), true)
                    SKGTESTERROR(QLatin1String("SUBOPE:setQuantity"), subop_1.setQuantity(((i * j) % 60) - 10), true)
                    SKGTESTERROR(QLatin1String("SUBOPE:setOrder"), subop_1.setOrder(i), true)
                    SKGTEST(QLatin1String("SUBOPE:getOrder"), subop_1.getOrder(), i)
                    SKGTESTERROR(QLatin1String("SUBOPE:save"), subop_1.save(), true)
                }
            }

            cats[0].merge(cats[1]);
            payees[0].merge(payees[1]);

            // Delete
            delete [] cats;
            delete [] modes;
            delete [] comments;
            delete [] payees;
        }  // A commit is done here because the scope is close

        SKGTESTERROR(QLatin1String("ACCOUNT:getCurrentAmount"), account.load(), true)
        SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), account.getCurrentAmount(), 140165)

        QFile(filename).remove();
        SKGTESTERROR(QLatin1String("DOC:saveAs"), document1.saveAs(filename), true)
    }

    // ============================================================================
    {
        SKGTraces::SKGPerfo = true;
        SKGTRACEIN(0, "openTest")

        // Test bank document
        SKGDocumentBank document1;
        {
            SKGTRACEIN(0, "openTest-Load")
            SKGTESTERROR(QLatin1String("document1.load"), document1.load(SKGTest::getTestPath(QLatin1String("OUT")) % "/skgtestbigdocument/skgtestbigdocument.skg"), true)
        }

        {
            SKGTRACEIN(0, "openTest-Get")
            SKGAccountObject account;
            SKGTESTERROR(QLatin1String("SKGAccountObject::getObjectByName"), SKGAccountObject::getObjectByName(&document1, QLatin1String("v_account"), QLatin1String("Courant steph"), account), true)
            SKGTEST(QLatin1String("ACCOUNT:getCurrentAmount"), account.getCurrentAmount(), 140165)

            SKGAccountObject::SKGListSKGObjectBase objects;
            SKGTESTERROR(QLatin1String("SKGAccountObject::getObjects"), document1.getObjects(QLatin1String("v_operation"), QLatin1String(""), objects), true)

            int nbobj = 0;
            SKGTESTERROR(QLatin1String("SKGAccountObject::getNbObjects"), document1.getNbObjects(QLatin1String("v_operation"), QLatin1String(""), nbobj), true)
        }

        {
            SKGTRACEIN(0, "openTest-Save")
            SKGTESTERROR(QLatin1String("document1.save"), document1.save(), true)
        }
    }

    // End test
    SKGENDTEST()
}
