/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtestmacro.h"
#include "skgimportexportmanager.h"
#include "skgbankincludes.h"


struct test {
    QString fileName;
    QString password;
    QMap<QString, double> expectedAccountAmount;
};

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc);
    Q_UNUSED(argv);

    // Init test
    SKGINITTEST(true);

    {
        // Test import MNY
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT_MNY", err);
            SKGImportExportManager impmissing(&document1, QUrl("missingfile.mny"));
            {
                QMap<QString, QString> params = impmissing.getImportParameters();
                params["install_sunriise"] = 'Y';
                impmissing.setImportParameters(params);
            }
            SKGTESTERROR("imp1.importFile", impmissing.importFile(), false);

            SKGImportExportManager impwrong(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportmny1/wrongfile.mny"));
            {
                QMap<QString, QString> params = impwrong.getImportParameters();
                params["install_sunriise"] = 'Y';
                impwrong.setImportParameters(params);
            }
            SKGTESTERROR("imp1.importFile", impwrong.importFile(), false);

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportmny1/money2004-pwd:123@ABC!.mny"));
            {
                QMap<QString, QString> params = imp1.getImportParameters();
                params["password"] = "wrong password";
                params["install_sunriise"] = 'Y';
                imp1.setImportParameters(params);
            }
            SKGTEST("imp1.importFile", imp1.importFile().getReturnCode(), ERR_ENCRYPTION);
        }
    }


    QList<test> listTests;
    {
        test t1;
        t1.fileName = "A B/money2002.mny";

        QMap<QString, double> accounts;
        accounts["None Investment (Cash)"] = 0.0;
        accounts["Investments to Watch"] = 0.0;
        accounts["None Investment"] = 1.49;
        t1.expectedAccountAmount = accounts;
        listTests << t1;
    }
    {
        test t1;
        t1.fileName = "money2001.mny";

        QMap<QString, double> accounts;
        accounts["Investments to Watch"] = 0.0;
        t1.expectedAccountAmount = accounts;
        listTests << t1;
    }
    {
        test t1;
        t1.fileName = "money2004-pwd:123@ABC!.mny";
        t1.password = "123@ABC!";

        QMap<QString, double> accounts;
        accounts["Investments to Watch"] = 0.0;
        t1.expectedAccountAmount = accounts;
        listTests << t1;
    }
    {
        test t1;
        t1.fileName = "money2005-pwd:123@ABC!.mny";
        t1.password = "123@ABC!";

        QMap<QString, double> accounts;
        accounts["George's Pension Plan"] = 250.0;
        accounts["Investments to Watch"] = 0.0;
        accounts["Invoice Sales Ledger"] = 1780.71;
        accounts["Stocks and Shares"] = 830.5;
        accounts["Stocks and Shares (Cash)"] = -178.0;
        accounts["Woodgrove Bank Credit Card"] = 1013.0;
        accounts["Woodgrove Bank Current"] = 20280.14;
        accounts["Woodgrove Bank Savings"] = 800.0;
        t1.expectedAccountAmount = accounts;
        listTests << t1;
    }
    {
        test t1;
        t1.fileName = "money2008-pwd:TEST12345.mny";
        t1.password = "TEST12345";

        QMap<QString, double> accounts;
        accounts["Investments to Watch"] = 0.0;
        t1.expectedAccountAmount = accounts;
        listTests << t1;
    }

    foreach(const test & t, listTests) {
        // Test import MNY
        SKGDocumentBank document1;
        SKGTESTERROR("document1.initialize()", document1.initialize(), true);
        SKGError err;
        {
            // Scope of the transaction
            SKGBEGINTRANSACTION(document1, "IMPORT_MNY", err);

            SKGImportExportManager imp1(&document1, QUrl::fromLocalFile(SKGTest::getTestPath("IN") % "/skgtestimportmny1/" % t.fileName));
            QMap<QString, QString> params = imp1.getImportParameters();
            params["password"] = t.password;
            params["install_sunriise"] = 'Y';
            imp1.setImportParameters(params);
            SKGTESTERROR(t.fileName % ".importFile", imp1.importFile(), true);
        }

        QStringList keys = t.expectedAccountAmount.keys();
        foreach(const QString & k, keys) {
            SKGAccountObject account(&document1);
            SKGTESTERROR(t.fileName % ".setName(" % k % ")", account.setName(k), true);
            SKGTESTERROR(t.fileName % ".load(" % k % ")", account.load(), true);
            SKGTEST(t.fileName % ".getCurrentAmount(" % k % ")", SKGServices::doubleToString(account.getCurrentAmount()), SKGServices::doubleToString(t.expectedAccountAmount[k]));
        }

        if (t.fileName == "money2005-pwd:123@ABC!.mny") {
            bool check = false;
            SKGTESTERROR(t.fileName % ".existObjects", document1.existObjects("v_operation", "t_TRANSFER='Y'", check), true);
            SKGTESTBOOL(t.fileName % ".existObjects", check, true);
        }
    }
    // End test
    SKGENDTEST();
}
