/////////////////////////////////////////////////////////////////////////////
//
// Plugins manager
//
// Time-stamp: <97/05/06 04:08:05 vels>
// Copyright (c) VDOnet Corp. 1996
//
/////////////////////////////////////////////////////////////////////////////

#include <unistd.h>
#include <stdlib.h>
#include <limits.h>
#include <stdio.h>

#include "kplugmng.h"
#include "kplugmng.moc"

// Constructor
KPluginManager::KPluginManager()
{
  m_pluginList.setAutoDelete(TRUE);
  m_chain.setAutoDelete(FALSE);
}

// Load input plugin (will be placed first in the chain)
KPlugin *KPluginManager::loadInputPlugin(const char *path)
{
  KPlugin *plugin;
  if (!(plugin=loadPlugin(path)))
    return NULL;

  // Check if it's input plugin
  if (plugin->type() != KPlugin::Input)
  {
    warning("KPluginManager::loadInputPlugin(\"%s\") not an input plugin",
            path);
    unloadPlugin(plugin);
    return NULL;
  }

  // If we already have input plugin - remove it
  KPlugin *first = m_chain.first();
  if (first && first->type() == KPlugin::Input)
  {
    unloadPlugin(first);
    m_chain.removeFirst();
  }
  
  // Insert new plugin
  m_chain.insert(0, plugin);
  return plugin;
}

// Load output plugin (will be placed last in the chain)
KPlugin *KPluginManager::loadOutputPlugin(const char *path)
{
  KPlugin *plugin;
  if (!(plugin=loadPlugin(path)))
    return NULL;

  // Check if it's output plugin
  if (plugin->type() != KPlugin::Output)
  {
    warning("KPluginManager::loadOutputPlugin(\"%s\") not an output plugin",
            path);
    unloadPlugin(plugin);
    return NULL;
  }

  // If we already have output plugin - remove it
  KPlugin *last = m_chain.last();
  if (last && last->type() == KPlugin::Output)
  {
    unloadPlugin(last);
    m_chain.removeLast();
  }
  
  // Append new plugin to the chain
  m_chain.append(plugin);
  return plugin;
}

// Load filter plugin
KPlugin *KPluginManager::loadFilterPlugin(const char *path, 
                                          const char *afterPath)
{
  KPlugin *plugin;
  if (!(plugin=loadPlugin(path)))
    return NULL;

  // Insert plugin before the last one
  if (!afterPath)
  {
    m_chain.insert(m_chain.count() - 1, plugin);
  }
  else // insert plugin after the 'afterPath' one
  {
    bool inserted = FALSE;

    // Search for plugin info item
    PluginInfo *iter;
    for (iter=m_pluginList.first(); iter != NULL; iter=m_pluginList.next())
    {
      // Found ?
      if (iter->path() == afterPath)
      {
        // Search for this plugin in chain
        int index = m_chain.findRef(m_pluginList.current()->plugin());
        if (index < 0)
        {
          warning("KPluginManager::loadFilterPlugin() oops.. this is bad !");
          unloadPlugin(plugin);
          return NULL;
        }

        // We can't insert filter before input plugin
        if (index == 0)
          index++;

        m_chain.insert(index, plugin);
        inserted = TRUE;
        break;
      }
    }

    if (!inserted)
    {
      warning("KPluginManager::loadFilterPlugin() plugin not found in chain");
      unloadPlugin(plugin);
      return NULL;
    }
  }

  return plugin;
}

// Let's rock'n roll !
int KPluginManager::start()
{
  KPlugin *first = m_chain.first();
  if (!first || first->type() != KPlugin::Input)
  {
    warning("KPluginManager::start() first plugin is not input plugin");
    return -1;
  }
  
  return first->start();
}

// Closing time...
void KPluginManager::stop()
{
  m_chain.first()->stop();
}

// Chain did it's job
void KPluginManager::chainFinished()
{
  emit finished();
}

// Load plugin given a name (if name is not a full path - we'll
// try to search for the plugin in standart places)
KPlugin *KPluginManager::loadPlugin(const char *path)
{
  // Construct path to plugin (MUST be full path)
  char plugPath[PATH_MAX];
  if (*path != '/')
  {
#if 0
    char *kdedir = getenv("KDEDIR");
    if (!kdedir)
    {
      warning("KDEDIR is not set !");
      return NULL;
    }

    strcpy(plugPath, kdedir);
    strcat(plugPath, "plugins/");

    // TODO:
    // Perform search in all subdirectories...
#else
    // debug
    getcwd(plugPath, PATH_MAX);
#endif
    strcat(plugPath, "/");
  }

  strcat(plugPath, path);

  // Load plugin module
  void *handle = dlopen(plugPath, RTLD_NOW | RTLD_GLOBAL);
  if (!handle)
  {
    warning("KPluginManager::loadPlugin(\"%s\") %s", plugPath, dlerror());
    return NULL;
  }

  // Load function that creates plugin
  KPlugin *(*initInstance)(void);
  const char *error;
  initInstance = (KPlugin *(*)(void))dlsym(handle, "initInstance");
  if ((error = dlerror()) != NULL)
  {
    warning("KPluginManager::loadPlugin(\"%s\") %s", plugPath, error);
    dlclose(handle);
    return NULL;
  }

  // Create plugin
  KPlugin *plugin = initInstance();
  plugin->setManager(this);
  
  // Add it to the 'registry' :)
  PluginInfo *pi = new PluginInfo(plugin, QString(plugPath), handle);
  m_pluginList.append(pi);

  return plugin;
}

// Unload plugin
int KPluginManager::unloadPlugin(KPlugin *plugin)
{
  bool removed = FALSE;

  // Search for plugin info item
  PluginInfo *iter;
  for (iter=m_pluginList.first(); iter != NULL; iter=m_pluginList.next())
  {
    // Remove it
    if (iter->plugin() == plugin)
    {
      m_pluginList.remove();
      removed = TRUE;
      break;
    }
  }

  // Not found ?
  if (!removed)
  {
    warning("KPluginManager::unloadPlugin() plugin not found");
    return -1;
  }

  return 0;
}

// Get input plugin
KPlugin *KPluginManager::inputPlugin()
{
  KPlugin *first = m_chain.first();
  if (!first || first->type() != KPlugin::Input)
  {
    warning("KPluginManager::inputPlugin() first plugin is not input plugin");
    return NULL;
  }
  
  return first;
}

// Get output plugin
KPlugin *KPluginManager::outputPlugin()
{
  KPlugin *last = m_chain.last();
  if (!last || last->type() != KPlugin::Output)
  {
    warning("KPluginManager::outputPlugin() last plugin is not output plugin");
    return NULL;
  }
  
  return last;
}

// Get next plugin in chain
KPlugin *KPluginManager::next(KPlugin *current)
{
  // Find given plugin
  int index = m_chain.findRef((const KPlugin *)current);
  if (index < 0)
  {
    warning("KPluginManager::next() passed plugin is not in the chain");
    return NULL;
  }

  // Get the next one
  KPlugin *next;
  if (!(next=m_chain.next()))
  {
    warning("KPluginManager::next() passed plugin is last in the chain");
    return NULL;
  }

  return next;
}
