/* This file is part of the KDE libraries
    Copyright (c) 1998 Emmeran Seehuber (the_emmy@hotmail.com)
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include "kldevice.h"
#include "qwidget.h"


/**
* KLDevice - Implementation
*/

void KLDevice::KLDeviceShowinfo::init()
{
  x = 0;
  y = 0;
  xSize = 0;
  ySize = 0;
  minXSize = 0;
  minYSize = 0;
  maxXSize = 0;
  maxYSize = 0;
  xOff = 0;
  yOff = 0;
}

KLDevice::KLDevice()
{
  a_rootChild = 0;
  config = 0;
  showed = false,
  setupdone = false;
  layoutStop = 0;
  doRelayout = FALSE;
  doingRelayout = FALSE;
  refreshStop = 0;
  doRefresh = FALSE;
  a_grapper = 0;
}

KLDevice::~KLDevice()
{
  hideDevice();
  if( a_rootChild ) 
    delete a_rootChild;
}


void KLDevice::grapDeviceInput(QObject *grapper)
{
  HASSERT( setupdone );
  HASSERT( a_grapper == NULL ); // Dont grap input twice !!!
  HASSERT( a_rootChild );

  a_grapper = grapper;
  getDeviceHandle()->installEventFilter(a_grapper);
  a_rootChild->setupGrap();
  a_oldAcceptDrops = getDeviceHandle()->acceptDrops();
  getDeviceHandle()->setAcceptDrops(true);
}

void KLDevice::releaseDeviceInput()
{
  HASSERT( a_grapper );
  a_rootChild->cleanupGrap();
  getDeviceHandle()->removeEventFilter(a_grapper);
  a_grapper = 0;
  getDeviceHandle()->setAcceptDrops(a_oldAcceptDrops);
}


void KLDevice::getRCSize(ulong &xSize, ulong &ySize) const
{
  if( a_rootChild && (a_rootChild->a_state == KLChild::SI_Showed) ) {
    xSize = a_rootChild->showInfo().xSize;
    ySize = a_rootChild->showInfo().ySize;
  }
  else {
    xSize = 0;
    ySize = 0;
  }
}


bool KLDevice::showDevice()
{
  if( showed ) { // We are already showed
    WARN("Showed twice !");
    return false;
  }

  if( createWindow() ) {
    if( calcWindowPos( &dsi ) ) {
      if( showWindow( &dsi ) ) 
        return TRUE;
      else
        WARN("Failed to display the dialog!\n");

      showed = false;
      setupdone = false;
    }
    else
      WARN("Can`t calculcate the window size!\n");

    deleteWindow();
  }
  else
    WARN("Failed to create window\n");

  return FALSE;
}


void KLDevice::hideDevice()
{
  if( showed ) 
    hideWindow();
  showed = false;

  if( setupdone ) 
    deleteWindow();
  setupdone = false;
}


void KLDevice::setDeviceSize(unsigned long xSize, unsigned long ySize)
{
  // WARN("Changing size ");
  HASSERT(showed);

  HASSERT(a_rootChild);

  KLChild::KLShowInfo &showInfo = a_rootChild->showInfo();
  if( showInfo.xSize != xSize || showInfo.ySize != ySize ) {
    // Set the should be sizes
    showInfo.xSize = xSize;
    showInfo.ySize = ySize;
  }
  else {
    // WARN("... no changes needed ");
  }


  // Let the this changes take effect
  relayout();

  // Are the given sizes illegal ?
  if( xSize != showInfo.xSize || ySize != showInfo.ySize ) {
    // WARN("... given sizes are illegal ! Relayouting takes process. \n");
    setDeviceSize(showInfo.xSize, showInfo.ySize);
  }

  // WARN("... done !\n");
}


void KLDevice::relayout()
{
  HASSERT( showed );
  if( layoutStop ) {
    doRelayout = true;
    return;
  }
  
  if( doingRelayout ) {
    // Relayout is already in progress
    return;
  }

  doingRelayout = true;
  // WARN("Starting relayout ... \n");

  KLChild::KLShowInfo &showInfo = a_rootChild->showInfo(); 
  ulong xSize = showInfo.xSize;
  ulong ySize = showInfo.ySize;

  // Hide widgets
  a_rootChild->klHide();

  // ReAsk sizes
  KLChild::KLMinMaxSizes &minMax = a_rootChild->minMaxSizes();
  minMax.init();
  a_rootChild->klDoAskMinMax( &minMax );

  // Check, if the old device sizes are now incorrect
  ulong xOldSize = xSize;
  ulong yOldSize = ySize;
  if( xSize < minMax.minX )
    xSize = minMax.minX;
  if( ySize < minMax.minY ) 
    ySize = minMax.minY;
  if( xSize > minMax.maxX )
    xSize = minMax.maxX;
  if( ySize > minMax.maxY )
    ySize = minMax.maxY;

  // Set the new sizes
  // .. on the window
  ulong oldXOff = dsi.xOff;
  ulong oldYOff = dsi.yOff;

  dsi.init();
  dsi.xSize = xSize;
  dsi.ySize = ySize;
  dsi.minXSize = minMax.minX;
  dsi.minYSize = minMax.minY;
  dsi.maxXSize = minMax.maxX;
  dsi.maxYSize = minMax.maxY;
  dsi.xOff = oldXOff;
  dsi.yOff = oldYOff;
  setLimits(&dsi);

  // On the widgets 
  showInfo.xSize = xSize;
  showInfo.ySize = ySize;
  if( xSize != xOldSize || ySize != yOldSize ) 
    setDeviceSize(showInfo.xSize, showInfo.ySize);
  a_rootChild->klShow(&showInfo);

  // Just refersh everything
  refreshDisplay();
  doingRelayout = false;
}

void KLDevice::stopRelayout()
{
  layoutStop++;
}

void KLDevice::startRelayout()
{
  HASSERT( layoutStop ) ;
  layoutStop--;
  if( !layoutStop && doRelayout ) {
    doRelayout = false;
    relayout();
  }
}


bool KLDevice::refreshDisplay()
{
  if( refreshStop ) {
    doRefresh = true;
    return false;
  }
  return true;
}

void KLDevice::stopRefresh()
{
  refreshStop++;
}

void KLDevice::startRefresh()
{
  HASSERT(refreshStop);
  refreshStop--;
  if( !refreshStop && doRefresh ) {
    doRefresh = FALSE;
    refreshDisplay();
  }
}


void KLDevice::setRootChild( KLChild *rootchild )
{
  HASSERT( setupdone == FALSE && showed == FALSE );
  a_rootChild = rootchild;
}


KLChild *KLDevice::rootChild()
{
  return a_rootChild;
}


void KLDevice::setConfig( KConfigBase *newconfig )
{
  config = newconfig;
}



bool KLDevice::calcWindowPos( KLDeviceShowinfo *deviceShowInfo )
{
  HASSERT( deviceShowInfo );
  HASSERT( a_rootChild );
  HASSERT( !setupdone );

  // Reset displayinfo
  deviceShowInfo->init();

  if( a_rootChild ) {

    // Setup controls
    KLChild::KLSetupInfo &setupInfo = a_rootChild->setupInfo();
    setupInfo.device = this;
    // HASSERT( config ); // Configuration must be set !!
    setupInfo.config = config;

    if( a_rootChild->klSetup( &setupInfo ) ) {
      setupdone = true;

      // Ask for sizes
      KLChild::KLMinMaxSizes &minMaxSizes = a_rootChild->minMaxSizes();
      minMaxSizes.init();
      a_rootChild->klDoAskMinMax(&minMaxSizes);
      
      // Set device sizes
      deviceShowInfo->xSize = minMaxSizes.defX;
      deviceShowInfo->ySize = minMaxSizes.defY;
      deviceShowInfo->minXSize = minMaxSizes.minX;
      deviceShowInfo->minYSize = minMaxSizes.minY;
      deviceShowInfo->maxXSize = minMaxSizes.maxX;
      deviceShowInfo->maxYSize = minMaxSizes.maxY;
      return true;
    }
    else
      WARN("Setup of childs failed !!\n");
  }
  return false;
}


bool KLDevice::showWindow( KLDeviceShowinfo *deviceShowInfo )
{
  HASSERT( !showed );
  HASSERT( a_rootChild );
  HASSERT( deviceShowInfo );

  if( showed )
    return false;

  // Fill out showinfo
  KLChild::KLShowInfo &showInfo = a_rootChild->showInfo();
  showInfo.x = deviceShowInfo->xOff;
  showInfo.y = deviceShowInfo->yOff;
  showInfo.xSize = deviceShowInfo->xSize;
  showInfo.ySize = deviceShowInfo->ySize;

  if( a_rootChild->klShow(&showInfo) ) {
    showed = true;
    setLimits(&dsi);
    return true;
  }
  else
    return false;
}

void KLDevice::calcRWSize(ulong &xSize, ulong &ySize) const
{
  xSize += getTopBorder() + getBottomBorder();
  ySize += getLeftBorder() + getRightBorder();
}

void KLDevice::calcDSize(ulong &xSize, ulong &ySize) const
{
  xSize -= getTopBorder() + getBottomBorder();
  ySize -= getLeftBorder() + getRightBorder();
}



void KLDevice::hideWindow()
{
  HASSERT( showed );
  HASSERT( a_rootChild );
  if( showed ) 
    a_rootChild->klHide();

  showed = false;
}

void KLDevice::deleteWindow()
{
  if( showed )
    hideWindow();

  if( setupdone )
    a_rootChild->klCleanup();
  
  setupdone = false;
}



