/* This file is part of the KDE libraries
    Copyright (c) 1998 Emmeran Seehuber (the_emmy@hotmail.com)
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#ifndef KLDESIGNER_H
#define KLDESIGNER_H

#include "klchildmeta.h"

/**
* This class makes you able to let the user edit any existing
* KLE-GUI. 
* 
* You must do the following steps:
* 
* <UL>
* <LI> Create a KLDesigner object </LI>
* <LI> Register all your controls </LI>
* <LI> Call @ref #editDevice</LI>
* </UL>
*/

class KLDesignerPrivate;
class KLDevice;
class KLDesigner {
  KLDesignerPrivate *data;
public:
  KLDesigner();
  ~KLDesigner();

  /**
  * Enum flags to specify palettes to show 
  */
  enum Palettes {
    /**
    * Main control window of the designer. Contains
    * buttons to show/hide all other palettes and to 
    * enter the userglobal designer setup.
    */
    Pal_Main = (1<<0),
    /**
    * Showes the controltree.
    */
    Pal_Controls = (1<<1),
    /**
    * Showes the properties of the actual selected child and
    * let edit them
    */
    Pal_Properties = (1<<2),
    /**
    * Trashcanwindow. The user can drop childs on it to remove them
    * from the GUI.
    */
    Pal_Trashcan = (1<<3),    
  };


  /**
  * Set the application environment for the creation of objects
  *
  * @param appEnv Application depending environment data
  */
  void setAppEnv( void *appEnv );

  /**
  * Shows the designer toolpalettes.
  *
  * @see #hideDesigner
  * @param showFlags Specifies, wich palettes to show. Specify ~0 to show all
  *        palettes.
  */
  void showDesigner(ulong showFlags = ~0);

  /**
  * Hides the designer toolpalettes
  *
  * @see #showDesigner
  */
  void hideDesigner();

  /**
  * Attaches a layout device to the designer.
  *
  * The designer will grap all input events from the 
  * device.
  *
  * @param dev Device to attach
  * @see #detachDevice
  */
  void attachDevice(KLDevice *dev);

  /**
  * Detach a layout device from the designer.
  *
  * The device will get the input back.
  * @see #attachDevice
  */
  void detachDevice(KLDevice *dev);

  /**
  * Use this list to registrate all your classes
  *
  * You also use this list for dumping/restoring
  * on devices.
  * 
  * The KLE common controls are already registered
  *
  * @return the designers registration list
  */
  KLMetaRegList *regList();
};


/**
* Descripes a dropmark
*/
class KLGroup;
class KLDropMark {
public:
  /**
  * Coords are in device coords 
  */
  ulong x;
  ulong y;
  ulong xSize;
  ulong ySize;
  
  enum DropAction { 
    DA_Append,
    DA_AddBefore,
    DA_Replace,
    DA_Delete
  } dropAction;

  KLGroup *groupToInsert;
  KLChild *toAddBefore; // If DropAction == DA_Replace, this is the child to be replaced
  KLChild *dontAccept1; // Dont accept this child, it is youre neigbour (when not 0)
  KLChild *dontAccept2;

public:
  /**
  * Execute the drop action
  *
  * This means, insert the child into the group
  *
  * @param child child to insert here
  */
  void doAction(KLChild *child);
};


/**
* List of dropmarks
*/
class KLDropMarkList : public QList<KLDropMark> {
public:
  /**
  * Adds a dropmark to this list. TODO 
  *
  * x and y is in device coords.
  */
  void addDropMark(ulong x, ulong y, ulong xSize, ulong ySize, KLDropMark::DropAction dropAction, 
                   KLGroup *groupToInsert, KLChild *toAddBefore, KLChild *dontAccept1, KLChild *dontAccept2);

  /**
  * Find nearest valid DropMark
  *
  * x and y is in device coords.
  *
  * @param toDrop Child to drop here (can be 0 for new childs)
  */
  KLDropMark *findDropMark(long x, long y, KLChild *toDrop);


private:
  enum { FDM_Tolerance = 50 };
  enum { NotInDistance = 10000 };

  long CalcDist( KLDropMark *dm, ulong x, ulong y );
  long CalcMinDirectDist(ulong x, ulong x1, ulong x2);
};

#endif 
