/* This file is part of the KDE libraries
    Copyright (c) 1998 Emmeran Seehuber (the_emmy@hotmail.com)
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include "klchild.h"
#include "kldevice.h"
#include "klgroup.h"
#include <ctype.h>
#include <stdlib.h>
#include "klhelper.h"

/**
* HASSERT
*/

void _hassert(const char *why, const char *file, ulong line)
{
  printf("HASSERT(%s) failed at %s:%ld\n", why, file, line );
  char cHelp = 0;
  while( cHelp != 'C' && cHelp != 'S' && cHelp != 'A' ) {
    printf("(C)ontinue, (S)top, (A)bort ?");
    cHelp = toupper(getc(stdin));
  }

  switch( cHelp ) {
    case 'C' : return;
    case 'S' : {
      printf( "Trowing aritmetic exception ...\n" );
      int i = 1;
      i--;
      i = 2 / i; 
      return;
    }; break;
    case 'A' : exit(20); 
  }
}


/**
* Implementation of KLChild::KLMinMaxSizes
*/

void KLChild::KLMinMaxSizes::init()
{
  minX = 0;
  minY = 0;
  defX = 0;
  defY = 0;
  maxX = 0;
  maxY = 0;
}

void KLChild::KLMinMaxSizes::validate()
{
  maxX = KLMax(maxX,minX);
  maxY = KLMax(maxY,minY);
  defX = KLMin(defX,maxX);
  defY = KLMin(defY,maxY);
  defX = KLMax(defX,minX);
  defY = KLMax(defY,minY);
}

KLChild::KLMinMaxSizes KLChild::KLMinMaxSizes::operator+(const KLMinMaxSizes &mms )
{
  KLMinMaxSizes mmsResult;
  mmsResult = *this;
  mmsResult += mms;
  return mmsResult;
}


KLChild::KLMinMaxSizes &KLChild::KLMinMaxSizes::operator +=(const KLMinMaxSizes &mms )
{
  this->minX += mms.minX;
  this->minY += mms.minY;
  this->defX += mms.defX;
  this->defY += mms.defY;
  this->maxX += mms.maxX;
  this->maxY += mms.maxY;
  return *this;
}


void KLChild::KLMinMaxSizes::dumpSize()
{
  printf( "MinX: %5ld MinY: %5ld\n", minX, minY );
  printf( "DefX: %5ld DefY: %5ld\n", defX, defY );
  printf( "MaxX: %5ld MaxY: %5ld\n", maxX, maxY );
}

/**
* Implementation of KLChild::KLShowInfo
*/

void KLChild::KLShowInfo::init()
{
  this->x = 0;
  this->y = 0;
  this->xSize = 0;
  this->ySize = 0;
}


/**
* Implementation of KLChild::MyChildMeta - Metaclass
*/
KLChild::MyChildMeta::MyChildMeta()                       
{                                                           
  className = "KLChild" ;                                  
  userClassName = "General UI element";
  parentMeta = 0l;
}

KLChild::MyChildMeta KLChild::metaClass;

/**
* KLChild Implementation
*/

KLChild::KLChild()
{
  a_stopRelayout = 0;
  a_doRelayout  = FALSE;
  a_hidden = FALSE;
  a_fixXSize = 0;
  a_fixYSize = 0;
  a_xWeight = KLDefWeight;
  a_yWeight = KLDefWeight;
  layoutFinal = FALSE;
  askingNonFixSizes = FALSE; 
  parentChild = NULL;
  a_state = SI_None;
  a_disapearLevel = 0;
  realMetaClass = NULL;
}

void KLChild::setHidden( bool hide )
{
  if( hide != a_hidden ) {
    a_hidden = hide;
    doRelayout(TRUE); 
  }
}

void KLChild::setFixXSize(ulong fixXSize)
{
  if( fixXSize != a_fixXSize ) {
    a_fixXSize = fixXSize;
    doRelayout(TRUE);
  }
}


void KLChild::setFixYSize(ulong fixYSize)
{
  if( fixYSize != a_fixYSize ) {
    a_fixYSize = fixYSize;
    doRelayout(TRUE);
  }
}


void KLChild::setFixSizes(ulong fixXSize, ulong fixYSize)
{
  if( ( fixXSize != a_fixXSize ) ||
      ( fixYSize != a_fixYSize ) ) {
    a_fixXSize = fixXSize;
    a_fixYSize = fixYSize;
    doRelayout(TRUE);
  }
}


void KLChild::setWeight(ulong weight)
{
  a_xWeight = weight;
  a_yWeight = weight;
  doRelayout(TRUE);
}


void KLChild::setXWeight(ulong xWeight)
{
  if( a_xWeight != xWeight ) {
    a_xWeight = xWeight;
    doRelayout(TRUE);
  }
}


void KLChild::setYWeight(ulong yWeight)
{
  if( a_yWeight != yWeight ) {
    a_yWeight = yWeight;
    doRelayout(TRUE);
  }
}


void KLChild::setDisapearLevel(ulong disapearLevel)
{
  a_disapearLevel = disapearLevel;
  doRelayout(true);
}


KLChild::~KLChild()
{
  HASSERT( a_state == SI_None );
}


bool KLChild::isAChild( KLChild *_child ) const
{
  return _child == this;
}

bool KLChild::klSetup( KLSetupInfo *setupInfo )
{
  HASSERT( setupInfo );
  HASSERT( a_state == SI_None );
  if( setupInfo != &a_setupInfo ) // if it is not our own setupinfo, just copy it
    a_setupInfo = *setupInfo;

  // Check for validalid setupinfo
  HASSERT( setupInfo->device );
  // HASSERT( setupInfo->config );

  a_state = SI_SetupDone;
  return TRUE; // Setup of baseclass is ok -- it just cannot fail !
}


bool KLChild::klAskMinMax(KLMinMaxSizes *minMaxSizes)
{
  HASSERT( minMaxSizes );
  HASSERT( a_state == SI_SetupDone || a_state == SI_Showed );

  // Check for fix sizes
  if( ( a_fixXSize || a_fixYSize ) && !askingNonFixSizes ) {
    askingNonFixSizes = TRUE;

    // Is one size not fix ?
    if( a_fixYSize == 0 || a_fixYSize == 0 ) {

      // Yes! Ask for the other dynamic size !
      KLMinMaxSizes minMax;
      this->klDoAskMinMax( &minMax );

      // Add the dynamic size
      if( a_fixXSize == 0 ) {
        minMaxSizes->minX += minMax.minX;
        minMaxSizes->defX += minMax.defX;
        minMaxSizes->maxX += minMax.maxX;
      }
      else {
        HASSERT( a_fixYSize == 0 );
        minMaxSizes->minY += minMax.minY;
        minMaxSizes->defY += minMax.defY;
        minMaxSizes->maxY += minMax.maxY;
      }
    }

    // Add fix sizes
    minMaxSizes->minX += a_fixXSize;
    minMaxSizes->defX += a_fixXSize;
    minMaxSizes->maxX += a_fixXSize;
    minMaxSizes->minY += a_fixYSize;
    minMaxSizes->defY += a_fixYSize;
    minMaxSizes->maxY += a_fixYSize;

    askingNonFixSizes = false;
    return FALSE; // Dont add any sizes future !
  } // if Fix sizes

  return TRUE; // We dont add fix sizes => let the subclass add there sizes
}


bool KLChild::klShow(KLShowInfo *showInfo)
{
  HASSERT( showInfo );
  HASSERT( a_state == SI_SetupDone );

  if( showInfo != &a_showInfo ) // if it is not our own showinfo, just copy it
    a_showInfo = *showInfo;

  // HASSERT correct sizes
  HASSERT( showInfo->xSize >= a_minMaxSizes.minX );
  HASSERT( showInfo->ySize >= a_minMaxSizes.minY );
  HASSERT( showInfo->xSize <= a_minMaxSizes.maxX );
  HASSERT( showInfo->ySize <= a_minMaxSizes.maxY );

  a_state = SI_Showed;

  return TRUE; // Baseclass show can not fail !
}


void KLChild::klHide()
{
  HASSERT( a_state == SI_Showed );
  a_state = SI_SetupDone;
}


void KLChild::klCleanup()
{
  HASSERT( a_state == SI_SetupDone );
  a_state = SI_None;
}


KLChild *KLChild::findChild( ulong x, ulong y ) const
{
  if( a_state == SI_Showed && a_showInfo.matchCoords( x, y ) )
    return( (KLChild *)this );
  return 0;
}


void KLChild::doRelayout( bool relayoutParent )
{
  // Are we not settet up ? => Dont try any relayout
  if( a_state == SI_None ) 
    return;

  // Only relayout, if showed or relayout of parent forced
  if( a_state != SI_Showed && !relayoutParent ) 
    return;

  // Is relayouting stoped ?
  if( a_stopRelayout )  {
    a_doRelayout = TRUE;
    return;
  }

  // Stop the redrawing of the UI
  HASSERT( a_setupInfo.device );
  a_setupInfo.device->stopRefresh();

  // Get the new sizes (and save the old one for compare)
  KLMinMaxSizes minMax = a_minMaxSizes;
  a_minMaxSizes.init();
  this->klDoAskMinMax(&a_minMaxSizes);

  // Have the sizes changed or is just a 
  // relayout of parent forced ?
  if( minMax.minX != a_minMaxSizes.minX ||
      minMax.minY != a_minMaxSizes.minY ||
      minMax.maxX != a_minMaxSizes.maxX ||
      minMax.maxY != a_minMaxSizes.maxY ||
      relayoutParent ) {
    if( parentChild ) // Is there a parent ?
      parentChild->doRelayout(FALSE); // Yes, let him do the relayout
    else
      a_setupInfo.device->relayout();
  }
  else {
    // Just hide/show us.
    klHide();
    klShow(&a_showInfo);
  }

  // Restart the redrawing of the UI
  a_setupInfo.device->startRefresh();
}


void KLChild::stopRelayout()
{
  a_stopRelayout++;
}


void KLChild::startRelayout()
{
  HASSERT( a_stopRelayout );
  a_stopRelayout--;
  if( !a_stopRelayout && a_doRelayout ) {
    a_doRelayout = FALSE;
    doRelayout(TRUE); // Now we force a relayout of the parent
  }
}

// Streaming operators
KLChild &KLChild::operator<<(KLChild::_child_change_func func)
{
  func(*this);
  return *this;
}

KLChild::KLStreamHelp &KLChild::operator<<(KLChild::_child_change_p1_func func)
{
  static KLStreamHelp sh;
  sh.child = this;
  sh.func = func;
  return sh;
}

KLChild &KLChild::KLStreamHelp::operator<<(ulong param)
{
  func(*child,param);
  return *child;
}

void KLChild::klPostAskMinMax(KLMinMaxSizes *minMaxSizes)
{
  // First: Validate all sizes
  if( a_xWeight == 0 )
    minMaxSizes->maxX = minMaxSizes->minX;
  if( a_yWeight == 0 )
    minMaxSizes->maxY = minMaxSizes->minY;
  minMaxSizes->validate();
}

void KLChild::klDoAskMinMax(KLMinMaxSizes *minMaxSizes)
{
  klAskMinMax(minMaxSizes);
  klPostAskMinMax(minMaxSizes);
}

// Metaclass stuff
KLChild *KLChild::MyChildMeta::createObject(void *) const
{
  HASSERT( false ); // Cant create an object of KLChild
  return NULL;
}

bool KLChild::MyChildMeta::dumpObject(KLChild *child, KLDumpDevice& dev) const
{
  QString helpStr;
  dev.writeEntry("hidden",child->hidden() ? "1" : "0" );
  dev.writeEntry("fixxsize",helpStr.setNum(child->fixXSize()));
  dev.writeEntry("fixysize",helpStr.setNum(child->fixYSize()));
  dev.writeEntry("xweight",helpStr.setNum(child->xWeight()));
  dev.writeEntry("yweight",helpStr.setNum(child->yWeight()));
  dev.writeEntry("disapearlevel",helpStr.setNum(child->disapearLevel()));
  return true; 
}

void KLChild::MyChildMeta::restoreObject(KLChild *child, KLDumpDevice& dev) const
{
  QString helpStr;
  child->setHidden(atol(dev.readEntry("hidden",child->hidden() ? "1" : "0" )) == 1);
  child->setFixXSize(atol(dev.readEntry("fixxsize",helpStr.setNum(child->fixXSize()))));
  child->setFixYSize(atol(dev.readEntry("fixysize",helpStr.setNum(child->fixYSize()))));
  child->setXWeight(atol(dev.readEntry("xweight",helpStr.setNum(child->xWeight()))));
  child->setYWeight(atol(dev.readEntry("yweight",helpStr.setNum(child->yWeight()))));
  child->setDisapearLevel(atol(dev.readEntry("disapearlevel",helpStr.setNum(child->disapearLevel()))));
}

bool KLChild::metaIsA(const char *classname)
{
  if( getRealMetaClass()->className == classname )
    return true;
  return false;
}

bool KLChild::metaInherits(const char *classname)
{
  const KLChildMeta *metaClass = getRealMetaClass();
  while( metaClass ) {
    if( metaClass->className == classname ) 
      return true;
    metaClass = metaClass->parentMeta;
  }
  return false;
}

const KLChildMeta *KLChild::getRealMetaClass() const
{
  if( realMetaClass )
    return realMetaClass;
  return getMetaClass();
}


void KLChild::setupGrap()
{
  // NOP
}


void KLChild::cleanupGrap()
{
  // NOP
}


void KLChild::addDropMarks( KLDropMarkList * ) const
{
  // NOP
}


/**
* Edit class for KLChild
*/

#include "klchildpriv.h"

KLChild *KLChild::MyChildMeta::createObjectEdit(KLChild *child,bool extended) const
{
  return new KLChildEditGroup(child, extended); 
}

#include "kledit.h"
#include "klcheckbox.h"
#include "kllabel.h"

KLChildEditGroup::KLChildEditGroup(KLChild *child,bool extended) : KLGridGroup(2,false)
{
  HASSERT(child);
  a_child = child;
  //KLCheckBox *hidden;
  KLLineEdit *fixxsize = 0;
  KLLineEdit *fixysize = 0;
  KLLineEdit *xweight = 0;
  KLLineEdit *yweight = 0;
  // KLLineEdit *disapear;
  /**this << new KLHVSpace()  not yet
        << (hidden = new KLCheckBox("hidden"));*/
  setWeight(400);
  if( extended ) {
    *this << (*(new KLLabel("Fix X Size")) << ::setWeight << 1)
          << (fixxsize = new KLLineEdit())
          << (*(new KLLabel("Fix Y Size")) << ::setWeight << 1)
          << (fixysize = new KLLineEdit());
  }
  *this << (*(new KLLabel("X-Weight")) << ::setWeight << 1)
        << (xweight = new KLLineEdit())
        << (*(new KLLabel("Y-Weight")) << ::setWeight << 1)
        << (yweight = new KLLineEdit());
        /*<< new KLLabel("Disapear Level") // Not yet implementated
        << (disapear = new KLLineEdit());*/
  QString helpStr;
  /*connect(hidden,SIGNAL(toggled(bool)),this,SLOT(togHidden(bool))); not yet
  hidden->setChecked(child->a_hidden);*/
  if( extended ) {
    connect(fixxsize,SIGNAL(textChanged(const char*)),this,SLOT(onFixXSize(const char*)));
    connect(fixysize,SIGNAL(textChanged(const char*)),this,SLOT(onFixYSize(const char*)));
    fixxsize->setText(helpStr.setNum(a_child->fixXSize()));
    fixysize->setText(helpStr.setNum(a_child->fixYSize()));
  }
  connect(xweight,SIGNAL(textChanged(const char*)),this,SLOT(onXWeight(const char*)));
  connect(yweight,SIGNAL(textChanged(const char*)),this,SLOT(onYWeight(const char*)));
  xweight->setText(helpStr.setNum(a_child->xWeight()));
  yweight->setText(helpStr.setNum(a_child->yWeight()));
}

void KLChildEditGroup::togHidden(bool hidden)
{
  a_child->setHidden(hidden);
}

void KLChildEditGroup::onFixXSize(const char*text)
{
  a_child->setFixXSize(atol(text));
}

void KLChildEditGroup::onFixYSize(const char*text)
{
  a_child->setFixYSize(atol(text));
}

void KLChildEditGroup::onXWeight(const char*text)
{
  a_child->setXWeight(atol(text));
}

void KLChildEditGroup::onYWeight(const char*text)
{
  a_child->setYWeight(atol(text));
}

#include "klchild.moc"

