/*
 * This file is part of telepathy-presence-applet
 *
 * Copyright (C) 2011 Francesco Nwokeka <francesco.nwokeka@gmail.com>
 * Copyright (C) 2011 David Edmundson <kde@davidedmundson.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

import Qt 4.7
import org.kde.plasma.core 0.1 as PlasmaCore
import org.kde.plasma.graphicswidgets 0.1 as PlasmaWidgets

Item {
    id: mainWidget;

    width: 250;
    height: 200;

    // used for plasmoid global object
    property int minimumWidth: 250;
    property int minimumHeight: 200;

    // keep track of GlobalPresence source
    PlasmaCore.DataSource {
        id: presenceSource;
        engine: "presence";
        connectedSources: sources;

        onSourceAdded: {
            if (source == "GlobalPresence") {
                connectSource(source);
            }
        }

        onDataChanged: {
            updatePopupIcon();
        }
    }

    // set popupIcon
    Component.onCompleted: {
        presenceSource.connectSource("GlobalPresence");
        plasmoid.aspectRatioMode = IgnoreAspectRatio;
        plasmoid.resize(250,200);
        setupContextMenu();
    }

    // Left panel
    LeftPanel {
        id: leftPanel;

        anchors.fill: parent;

        Component.onCompleted: {
            // set popupIcon with values loaded
            updatePopupIcon();
        }

        onSetGlobalStatusMessageSignal: {
            // pass current presence id and new global status message
            mainWidget.setGlobalPresence(leftPanel.globalPresence, leftPanel.globalPresenceMessage);
        }

        onSetGlobalNickSignal: {
            mainWidget.setGlobalNick(globalNick);
        }

        onSetGlobalAvatarSignal: {
            mainWidget.setGlobalAvatar(globalAvatar);
        }

        onSetGlobalPresenceSignal: {
            // set id and remember status message as well
            mainWidget.setGlobalPresence(globalPresenceIconName, leftPanel.globalPresenceMessage);
            updatePopupIcon();
        }
    }

    function setupContextMenu()
    {
        // global status
        plasmoid.setAction("contextOnline", i18n("Online"), "user-online");
        plasmoid.setAction("contextBusy", i18n("Busy"), "user-busy");
        plasmoid.setAction("contextAway", i18n("Away"), "user-away");
        plasmoid.setAction("contextAwayExtended", i18n("Extended Away"), "user-away-extended");
        plasmoid.setAction("contextInvisible", i18n("Invisible"), "user-invisible");
        plasmoid.setAction("contextOffline", i18n("Offline"), "user-offline");

        if (plasmoid.apiVersion >= 5) {
            plasmoid.setActionSeparator("seperator1");
        }

        plasmoid.setAction("contextAccountManager", i18n("Account manager"), "telepathy-kde");
        plasmoid.setAction("contextContactList", i18n("Contact list"), "meeting-attending");

        if (plasmoid.apiVersion >= 5) {
            plasmoid.setActionSeparator("seperator2");
        }

        // presence status actions
        plasmoid.action_contextOnline = function() {
            mainWidget.setGlobalPresence("online", leftPanel.globalPresenceMessage);
        }

        plasmoid.action_contextBusy = function() {
            mainWidget.setGlobalPresence("busy", leftPanel.globalPresenceMessage);
        }

        plasmoid.action_contextAway = function() {
            mainWidget.setGlobalPresence("away", leftPanel.globalPresenceMessage);
        }

        plasmoid.action_contextAwayExtended = function() {
            mainWidget.setGlobalPresence("away-extended", leftPanel.globalPresenceMessage);
        }

        plasmoid.action_contextInvisible = function() {
            mainWidget.setGlobalPresence("hidden", leftPanel.globalPresenceMessage);
        }

        plasmoid.action_contextOffline = function() {
            mainWidget.setGlobalPresence("offline", leftPanel.globalPresenceMessage);
        }

        // other actions
        plasmoid.action_contextContactList = function() {
            plasmoid.runCommand("telepathy-kde-contactlist");
        }

        plasmoid.action_contextAccountManager = function() {
            plasmoid.runApplication("kcm_telepathy_accounts");
        }
    }

    // updates popupicon to respect global status
    function updatePopupIcon()
    {
        console.log(presenceSource.data["GlobalPresence"]["currentPresence"]);
        if (presenceSource.data["GlobalPresence"]["currentPresence"] != leftPanel.globalPresence ||
             presenceSource.data["GlobalPresence"]["presenceMessage"] != leftPanel.globalPresenceMessage) {
            leftPanel.globalPresence = presenceSource.data["GlobalPresence"]["currentPresence"];
            leftPanel.globalPresenceMessage = presenceSource.data["GlobalPresence"]["presenceMessage"];
            plasmoid.popupIcon = QIcon("user-" + leftPanel.globalPresence);
        }

        console.log("UPDATED POPUP ICON To: " + leftPanel.globalPresence + " and " + leftPanel.globalPresenceMessage)
    }


    // set nick to all accounts
    function setGlobalNick(globalNick)
    {
        for (var i = 0; i < presenceSource.sources.length; ++i) {
            var sourceName = presenceSource.sources[i];
            var operation = presenceSource.serviceForSource(sourceName).operationDescription("setNickname");
            operation.nickname = globalNick;
            presenceSource.serviceForSource(sourceName).startOperationCall(operation);
        }
    }

    // set avatar to all accounts
    function setGlobalAvatar(globalAvatar)
    {
        for (var i = 0; i < presenceSource.sources.length; ++i) {
            var sourceName = presenceSource.sources[i];
            var operation = presenceSource.serviceForSource(sourceName).operationDescription("setAvatar");
            operation.url = globalAvatar;
            presenceSource.serviceForSource(sourceName).startOperationCall(operation);
        }

        // set globally as well so dataengine registers it
        var globalOperation = presenceSource.serviceForSource("GlobalPresence").operationDescription("setGlobalAvatar");
        globalOperation.avatar = globalAvatar;
        presenceSource.serviceForSource("GlobalPresence").startOperationCall(globalOperation);
    }

    // set presence to all accounts
    function setGlobalPresence(presenceType, presenceMessage)
    {
        console.log("WANT TO SET GLOBAL TO: " + presenceType + ":" + presenceMessage);

        var operation = presenceSource.serviceForSource("GlobalPresence").operationDescription("setGlobalPresence");
        operation.presence = presenceType;
        operation.presenceMessage = presenceMessage;
        presenceSource.serviceForSource("GlobalPresence").startOperationCall(operation);
    }  
    
}
