/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGBUDGETRULEOBJECTT_H
#define SKGBUDGETRULEOBJECTT_H
/** @file
 * This file defines classes SKGBudgetRuleObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "skgobjectbase.h"
#include "skgerror.h"
#include "skgcategoryobject.h"
#include "skgbankmodeler_export.h"

class SKGDocumentBank;

/**
 * This class is a budget rule
 */
class SKGBANKMODELER_EXPORT SKGBudgetRuleObject : public SKGObjectBase
{
    Q_OBJECT
public:
    /**
     * Condition
     */
    enum Condition {NEGATIVE = -1,
                    ALL = 0,
                    POSITIVE = 1
                   };
    /**
     * Condition
     */
    Q_ENUMS(Condition)

    /**
     * Mode
     */
    enum Mode {NEXT,
               CURRENT,
               YEAR
              };
    /**
     * Mode
     */
    Q_ENUMS(Mode)

    /**
    * Default constructor
    */
    explicit SKGBudgetRuleObject();

    /**
    * Constructor
    * @param iDocument the document containing the object
    * @param iID the identifier in @p iTable of the object
    */
    explicit SKGBudgetRuleObject(SKGDocument* iDocument, int iID = 0);

    /**
     * Destructor
     */
    virtual ~SKGBudgetRuleObject();

    /**
     * Copy constructor
     * @param iObject the object to copy
     */
    SKGBudgetRuleObject(const SKGBudgetRuleObject& iObject);

    /**
     * Copy constructor
     * @param iObject the object to copy
     */
    explicit SKGBudgetRuleObject(const SKGObjectBase& iObject);

    /**
     * Operator affectation
     * @param iObject the object to copy
     */
    virtual const SKGBudgetRuleObject& operator= (const SKGObjectBase& iObject) override;

    /**
     * Enable / disable condition on year of the budget
     * @param iEnable condition
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError enableYearCondition(bool iEnable);

    /**
     * To know if condition on year is enabled or disabled
     * @return condition
     */
    virtual int isYearConditionEnabled() const;

    /**
     * Set year of the condition of the rule
     * @param iYear the year (0=all year)
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setBudgetYear(int iYear);

    /**
     * Get year of the condition of the rule
     * @return year of the condition of the rule
     */
    virtual int getBudgetYear() const;

    /**
     * Enable / disable condition on month of the budget
     * @param iEnable condition
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError enableMonthCondition(bool iEnable);

    /**
     * To know if condition on month is enabled or disabled
     * @return condition
     */
    virtual int isMonthConditionEnabled() const;

    /**
     * Set month of the condition of the rule
     * @param iMonth the month
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setBudgetMonth(int iMonth);

    /**
     * Get month of the condition of the rule
     * @return month of the budget
     */
    virtual int getBudgetMonth() const;

    /**
     * Enable / disable condition on category of the budget
     * @param iEnable condition
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError enableCategoryCondition(bool iEnable);

    /**
     * To know if condition on category is enabled or disabled
     * @return condition
     */
    virtual int isCategoryConditionEnabled() const;

    /**
     * Set the category of the condition of the rule
     * @param iCategory the category
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setBudgetCategory(const SKGCategoryObject& iCategory);

    /**
     * Remove the category of the condition of the rule
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError removeBudgetCategory();

    /**
     * Get the category of the condition of the rule
     * @param oCategory the category
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError getBudgetCategory(SKGCategoryObject& oCategory) const;

    /**
     * Set the condition when the rule must be applied
     * @param iCondition the condition
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setCondition(SKGBudgetRuleObject::Condition iCondition);

    /**
     * Get the condition when the rule must be applied
     * @return the condition
     */
    virtual SKGBudgetRuleObject::Condition getCondition() const;

    /**
     * Set the quantity to transfer
     * @param iQuantity quantity
     * @param iAbsolute true means "an amount in primary unit", false means "a percentage"
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setQuantity(double iQuantity, bool iAbsolute);

    /**
     * Get quantity to transfer
     * @return quantity to transfer
     */
    virtual double getQuantity() const;

    /**
     * To know if the quantity is an amount or a percentage
     * @return true means "an amount in primary unit", false means "a percentage"
     */
    virtual bool isAbolute() const;

    /**
     * Enable / disable transfer change the category of the budget
     * @param iEnable condition
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError enableCategoryChange(bool iEnable);

    /**
     * To know if transfer change the category
     * @return transfer change the category
     */
    virtual int isCategoryChangeEnabled() const;

    /**
     * Set the transfer to apply
     * @param iMode the mode (NEXT=same category but for following budget, CURRENT=same period but for another category, YEAR=period of same year)
     * @param iCategory the category
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setTransfer(SKGBudgetRuleObject::Mode iMode, const SKGCategoryObject& iCategory = SKGCategoryObject());

    /**
     * Get the mode of the transfer
     * @return the mode
     */
    virtual SKGBudgetRuleObject::Mode getTransferMode() const;

    /**
     * Get the category of the transfer of the rule
     * @param oCategory the category
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError getTransferCategory(SKGCategoryObject& oCategory) const;

    /**
     * Process all rules
     * @param iDocument the document containing the object
     * @return an object managing the error
     *   @see SKGError
     */
    static SKGError processAllRules(SKGDocumentBank* iDocument);
};
/**
 * Declare the class
 */
Q_DECLARE_TYPEINFO(SKGBudgetRuleObject, Q_MOVABLE_TYPE);
#endif
