/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a delegate for budget.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgbudgetdelegate.h"
#include "skgobjectmodelbase.h"
#include "skgtraces.h"
#include "skgprogressbar.h"

#include <kcolorscheme.h>

#include <QSortFilterProxyModel>
#include <QPainter>

SKGBudgetDelegate::SKGBudgetDelegate(QObject* parent, SKGDocument* iDoc) : QStyledItemDelegate(parent), m_document(iDoc)
{}

SKGBudgetDelegate::~SKGBudgetDelegate()
{
    m_document = NULL;
}

void SKGBudgetDelegate::paint(QPainter* painter, const QStyleOptionViewItem& option, const QModelIndex& index) const
{
    if (!index.isValid()) return;
    SKGObjectModelBase* m = qobject_cast<SKGObjectModelBase*> ((QAbstractItemModel*) index.model());
    QSortFilterProxyModel* p = qobject_cast<QSortFilterProxyModel*> ((QAbstractItemModel*) index.model());
    if (p) m = (SKGObjectModelBase*) p->sourceModel();
    if (m) {
        QString att = m->getAttribute(index.column());

        //Compute pourcent of time
        QModelIndex idxs = (p ? p->mapToSource(index) : index);
        SKGObjectBase obj = m->getObject(idxs);

        int year = SKGServices::stringToDouble(obj.getAttribute("i_year"));
        int month = SKGServices::stringToDouble(obj.getAttribute("i_month"));
        QDate today = QDate::currentDate();
        double pourcent = 1;
        bool actif = true;
        if (year == today.year()) {
            if (month == 0) {
                QDate d1(year, 1, 1);
                pourcent = ((double) d1.daysTo(today)) / ((double) today.daysInYear());
            } else if (month == today.month()) {
                QDate d1(year, month, 1);
                pourcent = ((double) d1.daysTo(today)) / ((double) today.daysInMonth());
            } else if (month > today.month()) {
                pourcent = 0;
                actif = false;
            }
        } else if (year > today.year())     {
            pourcent = 0;
            actif = false;
        }

        if ((att == "f_budgeted" || att == "f_budgeted_modified") && actif) {
            double budgeted = SKGServices::stringToDouble(obj.getAttribute(att));
            double amount = SKGServices::stringToDouble(obj.getAttribute("f_CURRENTAMOUNT"));

            KColorScheme scheme(QPalette::Normal);
            QColor negativeC = scheme.foreground(KColorScheme::NegativeText).color().toHsv();
            QColor positiveC = scheme.foreground(KColorScheme::PositiveText).color().toHsv();
            QColor backC = scheme.foreground(KColorScheme::LinkText).color().toHsv();

            double coef = 0.3;
            negativeC.setHsv(negativeC.hue(), negativeC.saturation()*coef, negativeC.value());
            positiveC.setHsv(positiveC.hue(), positiveC.saturation()*coef, positiveC.value());
            backC.setHsv(backC.hue(), backC.saturation()*coef, backC.value());

            QBrush negative(negativeC);
            QBrush positive(positiveC);
            QBrush back(backC);

            painter->save();
            painter->setRenderHint(QPainter::Antialiasing);

            QStyleOptionViewItemV4 opt = option;
            QStyledItemDelegate::initStyleOption(&opt, index);
            QRect rect = opt.rect.adjusted(1, 1, -1, -1);

            // handle selection
            if (option.state & QStyle::State_Selected) {
                KStatefulBrush sb(KColorScheme::View, KColorScheme::NormalBackground);

                QBrush selectionBrush(sb.brush(QPalette::Active));
                painter->setBrush(selectionBrush);
                painter->drawRect(rect);
            }

            rect = opt.rect.adjusted(1, 1, -1, -1);

            painter->setPen(Qt::NoPen);

            if (budgeted > 0) {
                if (amount < 0) amount = 0;
                //Income
                if (amount < budgeted) {
                    //Draw red zone
                    painter->setBrush(negative);
                    painter->drawRect(rect);

                    //Draw blue zone
                    painter->setBrush(back);
                    QRect r2(rect.left(), rect.top(), rect.width() * (budgeted == 0 ? 1 : amount / budgeted), rect.height());
                    painter->drawRect(r2);
                } else {
                    //Draw green zone
                    painter->setBrush(positive);
                    painter->drawRect(rect);

                    //Draw blue zone
                    painter->setBrush(back);
                    QRect r2(rect.left(), rect.top(), rect.width() * (amount == 0 ? 0 : budgeted / amount), rect.height());
                    painter->drawRect(r2);
                }
            } else {
                if (amount > 0) amount = 0;
                //Expenditure
                if (amount < budgeted) {
                    //Draw red zone
                    painter->setBrush(negative);
                    painter->drawRect(rect);

                    //Draw blue zone
                    painter->setBrush(back);
                    QRect r2(rect.left(), rect.top(), rect.width() * (amount == 0 ? 0 : budgeted / amount), rect.height());
                    painter->drawRect(r2);
                } else {
                    //Draw green zone
                    painter->setBrush(positive);
                    painter->drawRect(rect);

                    //Draw blue zone
                    painter->setBrush(back);
                    QRect r2(rect.left(), rect.top(), rect.width() * (budgeted == 0 ? 1 : amount / budgeted), rect.height());
                    painter->drawRect(r2);
                }
            }

            //Draw time progress
            painter->setPen(Qt::black);
            QLine r2(rect.left() + rect.width()*pourcent, rect.top() + 1, rect.left() + rect.width()*pourcent, rect.top() + rect.height() - 1);
            painter->drawLine(r2);

            //Draw text
            painter->setPen(m->data(idxs, Qt::TextColorRole).value<QColor>());

            QTextOption to;
            to.setAlignment((Qt::AlignmentFlag) m->data(idxs, Qt::TextAlignmentRole).toInt());
            painter->drawText(rect, m->data(idxs).toString(), to);

            painter->restore();
            return;
        }
    }
    QStyledItemDelegate::paint(painter, option, index);
}

#include "skgbudgetdelegate.moc"
