/*
    SPDX-FileCopyrightText: 2018 Volker Krause <vkrause@kde.org>

    SPDX-License-Identifier: LGPL-2.0-or-later
*/

import QtCore
import QtQuick
import QtQuick.Layouts
import QtQuick.Controls as QQC2
import QtQuick.Dialogs as Dialogs
import org.kde.kirigami as Kirigami
import org.kde.kpublictransport
import org.kde.example

Kirigami.ApplicationWindow {
    title: "Journey Query"
    width: 640
    height: 800

    pageStack.initialPage: journyQueryPage

    Manager {
        id: ptMgr;
    }
    Settings {
        id: settings
        property alias allowInsecureBackends: ptMgr.allowInsecureBackends
        property alias enabledBackends: ptMgr.enabledBackends
        property alias disabledBackends: ptMgr.disabledBackends
    }

    JourneyQueryModel {
        id: journeyModel
        manager: ptMgr
    }

    JourneyTitleModel {
        id: titleModel
        sourceModel: journeyModel
    }

    globalDrawer: Kirigami.GlobalDrawer {
        actions: [
            Kirigami.Action {
                text: "Save..."
                icon.name: "document-save"
                onTriggered: fileDialog.open();
            },
            Kirigami.Action {
                icon.name: "help-about-symbolic"
                text: "Current Data Sources"
                enabled: journeyModel.attributions.length > 0
                onTriggered: {
                    aboutSheet.attributions = Qt.binding(function() { return journeyModel.attributions; });
                    aboutSheet.open();
                }
            },
            Kirigami.Action {
                icon.name: "help-about-symbolic"
                text: "All Data Sources"
                onTriggered: {
                    aboutSheet.attributions = Qt.binding(function() { return ptMgr.attributions; });
                    aboutSheet.open();
                }
            },
            Kirigami.Action {
                icon.name: "settings-configure"
                text: "Backends"
                onTriggered: pageStack.push(backendPage)
            }
        ]
    }

    Dialogs.FileDialog {
        id: fileDialog
        title: "Save Journey Data"
        fileMode: Dialogs.FileDialog.SaveFile
        nameFilters: ["JSON files (*.json)"]
        onAccepted: ExampleUtil.saveTo(journeyModel, fileDialog.selectedFile);
    }

    TestLocationsModel { id: exampleModel }
    AttributionSheet { id: aboutSheet }
    LocationDetailsSheet { id:locationDetailsSheet }

    function displayDuration(dur)
    {
        if (dur < 60)
            return "<1min";
        if (dur < 3600)
            return Math.floor(dur/60) + "min";
        return Math.floor(dur/3600) + ":" + Math.floor((dur % 3600)/60)
    }

    function displayDistance(dist)
    {
        if (dist == 0)
            return "";
        if (dist < 1000)
            return dist + "m";
        return Math.floor(dist/1000) + "km";
    }

    function locationName(loc)
    {
        switch(loc.type) {
            case Location.Stop: return "🚏 " + loc.name;
            case Location.RentedVehicleStation: return "🚏🚲 " + loc.name;
            case Location.RentedVehicle: return "🚲 " + loc.name;
            case Location.Place: return loc.name;
        }
    }

    Component {
        id: vehicleLayoutPage
        VehicleLayoutPage {
            publicTransportManager: ptMgr
        }
    }

    Component {
        id: indoorMapPage
        IndoorMapPage {}
    }

    Component {
        id: journeyDelegate
        QQC2.ItemDelegate {
            enabled: modelData.disruptionEffect != Disruption.NoService
            highlighted: false
            width: ListView.view.width
            contentItem: RowLayout {
                id: topLayout

                Kirigami.Icon {
                    id: icon
                    source: modelData.route.line.hasLogo ? modelData.route.line.logo : modelData.route.line.modeLogo
                    width: height
                    height: Kirigami.Units.iconSizes.large
                    visible: source != ""
                }

                Rectangle {
                    id: colorBar
                    width: Kirigami.Units.largeSpacing
                    color: modelData.route.line.hasColor ? modelData.route.line.color : "transparent"
                    Layout.fillHeight: true
                    visible: icon.source == ""
                }

                QQC2.Label {
                    text: {
                        switch (modelData.mode) {
                            case JourneySection.PublicTransport:
                            {
                                switch (modelData.route.line.mode) {
                                    case Line.Air: return "✈️";
                                    case Line.Boat: return "🛥️";
                                    case Line.Bus: return "🚍";
                                    case Line.BusRapidTransit: return "🚌";
                                    case Line.Coach: return "🚌";
                                    case Line.Ferry: return "⛴️";
                                    case Line.Funicular: return "🚞";
                                    case Line.LocalTrain: return "🚆";
                                    case Line.LongDistanceTrain: return "🚄";
                                    case Line.Metro: return "🚇";
                                    case Line.RailShuttle: return "🚅";
                                    case Line.RapidTransit: return "🚊";
                                    case Line.Shuttle: return "🚐";
                                    case Line.Taxi: return "🚕";
                                    case Line.Train: return "🚆";
                                    case Line.Tramway: return "🚈";
                                    case Line.RideShare: return "🚗";
                                    default: return "?";
                                }
                                break;
                            }
                            case JourneySection.Walking: return "🚶";
                            case JourneySection.Waiting: return "⌛";
                            case JourneySection.Transfer: return "⇄";
                            case JourneySection.RentedVehicle:
                            {
                                switch (modelData.rentalVehicle.type) {
                                    case RentalVehicle.Bicycle: return "🔑🚲";
                                    case RentalVehicle.Pedelec: return "🔑🔌🚲";
                                    case RentalVehicle.ElectricKickScooter: return "🔑🛴";
                                    case RentalVehicle.ElectricMoped: return "🔑🛵";
                                    case RentalVehicle.Car: return "🔑🚗";
                                    default: return "?";
                                }
                            }
                            case JourneySection.IndividualTransport:
                            {
                                switch (modelData.individualTransport.mode) {
                                    case IndividualTransport.Walk: return "🚶";
                                    case IndividualTransport.Bike: return "🚲";
                                    case IndividualTransport.Car: return "🚗";
                                }
                            }
                            default: return "?";
                        }
                    }
                    font.pointSize: Kirigami.Theme.defaultFont.pointSize * 2
                    visible: icon.source == ""
                }

                ColumnLayout {
                    Layout.fillWidth: true
                    RowLayout {
                        QQC2.Label {
                            text: "From: <a href=\"#from\">" + locationName(modelData.from) + "</a> Platform: " + modelData.scheduledDeparturePlatform
                            onLinkActivated: {
                                locationDetailsSheet.location = modelData.from;
                                locationDetailsSheet.open();
                            }
                        }
                        QQC2.Label {
                            text: modelData.expectedDeparturePlatform
                            color: modelData.departurePlatformChanged ? Kirigami.Theme.negativeTextColor : Kirigami.Theme.positiveTextColor
                            visible: modelData.hasExpectedDeparturePlatform
                        }
                    }
                    RowLayout {
                        QQC2.Label {
                            text: "Departure: " + modelData.scheduledDepartureTime.toTimeString()
                        }
                        QQC2.Label {
                            text: (modelData.departureDelay >= 0 ? "+" : "") + modelData.departureDelay
                            color: modelData.departureDelay > 1 ? Kirigami.Theme.negativeTextColor : Kirigami.Theme.positiveTextColor
                            visible: modelData.hasExpectedDepartureTime
                        }
                        QQC2.Label {
                            text: "<a href=\"#layout\">vehicle</a>"
                            visible: modelData.route.line.mode == Line.LongDistanceTrain || modelData.route.line.mode == Line.Train || modelData.route.name !== ""
                            onLinkActivated: applicationWindow().pageStack.push(vehicleLayoutPage, {"departure": modelData.departure });
                            Layout.fillWidth: true
                            horizontalAlignment: Text.Right
                        }
                    }
                    QQC2.Label {
                        Layout.fillWidth: true
                        text: {
                            switch (modelData.mode) {
                            case JourneySection.PublicTransport:
                            {
                                if (modelData.route.name !== "") {
                                    return modelData.route.line.modeString + " " + modelData.route.line.name + " (" + modelData.route.name
                                        + ") " + displayDuration(modelData.duration) + " / " + displayDistance(modelData.distance)
                                }
                                return modelData.route.line.modeString + " " + modelData.route.line.name + " " + displayDuration(modelData.duration) + " / " + displayDistance(modelData.distance)
                            }
                            case JourneySection.Walking:
                                return "Walk " + displayDuration(modelData.duration) + " / " + displayDistance(modelData.distance)
                            case JourneySection.Transfer:
                                return "Transfer " + displayDuration(modelData.duration)  + " / " + displayDistance(modelData.distance)
                            case JourneySection.Waiting:
                                return "Wait " + displayDuration(modelData.duration)
                            case JourneySection.RentedVehicle:
                                return "Drive (" + modelData.rentalVehicle.network.name + ") " + displayDuration(modelData.duration)  + " / " + displayDistance(modelData.distance);
                            case JourneySection.IndividualTransport:
                                return "Drive " + displayDuration(modelData.duration) + " / " + displayDistance(modelData.distance)
                            return "???";
                        }}
                    }
                    RowLayout {
                        QQC2.Label {
                            text: "To: <a href=\"#to\">" + locationName(modelData.to) + "</a> Platform: " + modelData.scheduledArrivalPlatform
                            onLinkActivated: {
                                locationDetailsSheet.location = modelData.to;
                                locationDetailsSheet.open();
                            }
                        }
                        QQC2.Label {
                            text: modelData.expectedArrivalPlatform
                            color: modelData.arrivalPlatformChanged ? Kirigami.Theme.negativeTextColor : Kirigami.Theme.positiveTextColor
                            visible: modelData.hasExpectedArrivalPlatform
                        }
                    }
                    RowLayout {
                        QQC2.Label {
                            text: "Arrival: " + modelData.scheduledArrivalTime.toTimeString()
                        }
                        QQC2.Label {
                            text: (modelData.arrivalDelay >= 0 ? "+" : "") + modelData.arrivalDelay
                            color: modelData.arrivalDelay > 1 ? Kirigami.Theme.negativeTextColor : Kirigami.Theme.positiveTextColor
                            visible: modelData.hasExpectedArrivalTime
                        }
                        QQC2.Label {
                            text: "<a href=\"#layout\">vehicle</a>"
                            visible: modelData.route.line.mode == Line.LongDistanceTrain
                            onLinkActivated: applicationWindow().pageStack.push(vehicleLayoutPage, {"departure": modelData.arrival });
                            Layout.fillWidth: true
                            horizontalAlignment: Text.Right
                        }
                    }
                    RowLayout {
                        visible: modelData.loadInformation.length > 0
                        QQC2.Label {
                            text: "Load: ";
                        }
                        Repeater {
                            model: modelData.loadInformation
                            RowLayout {
                                QQC2.Label {
                                    text: {
                                        switch (modelData.load) {
                                            case Load.Low: return "Low";
                                            case Load.Medium: return "Medium";
                                            case Load.High: return "High";
                                            case Load.Full: return "Full";
                                            default: return "?"
                                        }
                                    }
                                    color: {
                                        switch (modelData.load) {
                                            case Load.Low: return Kirigami.Theme.positiveTextColor;
                                            case Load.Medium: return Kirigami.Theme.neutralTextColor;
                                            case Load.High:
                                            case Load.Full:
                                                return Kirigami.Theme.negativeTextColor;
                                            default:
                                                return Kirigami.Theme.textColor;
                                        }
                                    }
                                }
                                QQC2.Label {
                                    text: "(class " + modelData.seatingClass + ")"
                                    visible: modelData.seatingClass != ""
                                }
                            }
                        }
                    }
                    QQC2.Label {
                        text: modelData.notes.join("<br/>")
                        textFormat: Text.RichText
                        visible: modelData.notes.length > 0
                        font.italic: true
                    }
                }
            }
            onClicked: {
                if (modelData.mode == JourneySection.PublicTransport) {
                    applicationWindow().pageStack.push(journeySectionPage, {"journeySection": modelData});
                } else if (modelData.path.sectionCount > 1) {
                    applicationWindow().pageStack.push(pathPage, {"path": modelData.path});
                }
            }
        }
    }

    Component {
        id: backendPage
        BackendPage {
            publicTransportManager: ptMgr
        }
    }

    Component {
        id: journeySectionPage
        JourneySectionPage {}
    }

    Component {
        id: pathPage
        PathPage {}
    }

    readonly property var individualTransportModes: [
        [{ mode: IndividualTransport.Walk }],
        [{ mode: IndividualTransport.Bike }],
        [{ mode: IndividualTransport.Bike, qualifier: IndividualTransport.Park }],
        [{ mode: IndividualTransport.Bike, qualifier: IndividualTransport.Rent }],
        [{ mode: IndividualTransport.Car, qualifier: IndividualTransport.Park }],
        [{ mode: IndividualTransport.Car, qualifier: IndividualTransport.Rent }],
        [{ mode: IndividualTransport.Car, qualifier: IndividualTransport.Pickup }],
        [{ mode: IndividualTransport.Car, qualifier: IndividualTransport.Dropoff }],
    ]
    ListModel {
        id: individualTransportModesModel
        ListElement { name: "Walk" }
        ListElement { name: "Take Bike" }
        ListElement { name: "Park Bike" }
        ListElement { name: "Rent Bike" }
        ListElement { name: "Park Car" }
        ListElement { name: "Rent Car" }
        ListElement { name: "Pickup by car" }
        ListElement { name: "Drop-off by car" }
    }

    Component {
        id: journyQueryPage
        Kirigami.Page {
            Settings {
                id: settings
                property alias singleBackend: backendBox.checked
                property alias backend: backendSelector.currentIndex
                property alias maxResults: maxResults.text
                property alias includeIntermediateStops: intermediateStops.checked
                property alias includePaths: includePaths.checked
                property alias accessMode: accessMode.currentIndex
                property alias egressMode: egressMode.currentIndex
            }

            ColumnLayout {
                anchors.fill: parent
                QQC2.CheckBox {
                    id: searchDirection
                    text: checked ? "Time is arrival" : "Time is departure"
                }

                QQC2.CheckBox {
                    text: "Allow insecure backends"
                    checked: ptMgr.allowInsecureBackends
                    onToggled: ptMgr.allowInsecureBackends = checked
                }

                RowLayout {
                    QQC2.CheckBox {
                        id: backendBox
                        text: "Select Backend:"
                    }
                    QQC2.ComboBox {
                        id: backendSelector
                        Layout.fillWidth: true
                        textRole: "identifier"
                        model: BackendModel {
                            manager: ptMgr
                        }
                        enabled: backendBox.checked
                    }
                }

                RowLayout {
                    QQC2.CheckBox {
                        id: ptMode
                        checked: true
                        text: "Public Transport"
                    }
                    QQC2.CheckBox {
                        id: rentalMode
                        checked: true
                        text: "Rental Vehicles"
                    }
                }
                RowLayout {
                    QQC2.Label { text: "Results:" }
                    QQC2.TextField {
                        id: maxResults
                        text: "10"
                    }
                    QQC2.CheckBox {
                        id: intermediateStops
                        checked: true
                        text: "Intermediate stops"
                    }
                    QQC2.CheckBox {
                        id: includePaths
                        text: "Paths"
                    }
                }

                QQC2.ComboBox {
                    id: accessMode
                    Layout.fillWidth: true
                    model: individualTransportModesModel
                    textRole: "name"
                }
                QQC2.ComboBox {
                    id: fromSelector
                    Layout.fillWidth: true
                    model: exampleModel
                    textRole: "label"
                    onCurrentIndexChanged: {
                        var obj = exampleModel.get(currentIndex);
                        fromName.text = obj.name == "" ? obj.label : obj.name;
                        fromLon.text = obj.lon;
                        fromLat.text = obj.lat;
                        if (toSelector.currentIndex == currentIndex) {
                            toSelector.currentIndex = (currentIndex + 1) % count;
                        }
                    }
                }
                RowLayout {
                    QQC2.TextField {
                        id: fromName
                    }
                    QQC2.TextField {
                        id: fromLon
                    }
                    QQC2.TextField {
                        id: fromLat
                    }
                }

                QQC2.ComboBox {
                    id: toSelector
                    Layout.fillWidth: true
                    model: exampleModel
                    textRole: "label"
                    onCurrentIndexChanged: {
                        var obj = exampleModel.get(currentIndex);
                        toName.text = obj.name == "" ? obj.label : obj.name;
                        toLon.text = obj.lon;
                        toLat.text = obj.lat;
                        if (fromSelector.currentIndex == currentIndex) {
                            fromSelector.currentIndex = (currentIndex - 1 + count) % count;
                        }
                    }
                }
                RowLayout {
                    QQC2.TextField {
                        id: toName
                    }
                    QQC2.TextField {
                        id: toLon
                    }
                    QQC2.TextField {
                        id: toLat
                    }
                }
                QQC2.ComboBox {
                    id: egressMode
                    Layout.fillWidth: true
                    model: individualTransportModesModel
                    textRole: "name"
                }
                QQC2.ComboBox {
                    id: lineModeSelector
                    Layout.fillWidth: true
                    model: [ "All", "Only long distance", "Local public transport", "Local trains only", "Rapit transit/metro/tram", "Bus" ]
                    property var currentMode: {
                        switch (currentIndex) {
                            case 1: return [Line.LongDistanceTrain, Line.Train];
                            case 2: return [Line.LocalTrain, Line.RapidTransit, Line.Metro, Line.Tramway, Line.Funicular, Line.Bus];
                            case 3: return [Line.LocalTrain];
                            case 4: return [Line.RapidTransit, Line.Metro, Line.Tramway, Line.Funicular];
                            case 5: return [Line.Bus];
                        }
                        return [];
                    }
                }

                RowLayout {
                    function setupRequestCommon()
                    {
                        journeyModel.request.dateTimeMode = searchDirection.checked ? JourneyRequest.Arrival : JourneyRequest.Departure;
                        journeyModel.request.dateTime = new Date(new Date().getTime() + (searchDirection.checked ? 7200000 : 0));
                        journeyModel.request.backends = backendBox.checked ? [ backendSelector.currentText ] : [];
                        journeyModel.request.downloadAssets = true
                        journeyModel.request.modes = (ptMode.checked ?  JourneySection.PublicTransport : JourneySection.Invalid)
                            | (rentalMode.checked ? JourneySection.RentedVehicle : JourneySection.Invalid);
                        journeyModel.request.maximumResults = maxResults.text;
                        journeyModel.request.includeIntermediateStops = intermediateStops.checked;
                        journeyModel.request.includePaths = includePaths.checked;
                        journeyModel.request.accessModes = individualTransportModes[accessMode.currentIndex];
                        journeyModel.request.egressModes = individualTransportModes[egressMode.currentIndex];
                        journeyModel.request.lineModes = lineModeSelector.currentMode;
                    }

                    QQC2.Button {
                        text: "Query"
                        onClicked: {
                            var from = journeyModel.request.from;
                            from.name = fromName.text;
                            from.latitude = fromLat.text;
                            from.longitude = fromLon.text;
                            journeyModel.request.from = from;
                            var to = journeyModel.request.to;
                            to.name = toName.text;
                            to.latitude = toLat.text;
                            to.longitude = toLon.text;
                            journeyModel.request.to = to;
                            parent.setupRequestCommon();
                        }
                    }
                    QQC2.Button {
                        text: "Query Name"
                        onClicked: {
                            var from = journeyModel.request.from;
                            from.name = fromName.text;
                            from.latitude = NaN;
                            from.longitude = NaN;
                            journeyModel.request.from = from;
                            var to = journeyModel.request.to;
                            to.name = toName.text;
                            to.latitude = NaN;
                            to.longitude = NaN;
                            journeyModel.request.to = to;
                            parent.setupRequestCommon();
                        }
                    }
                    QQC2.Button {
                        text: "Query Coord"
                        onClicked: {
                            var from = journeyModel.request.from;
                            from.name = "";
                            from.latitude = fromLat.text;
                            from.longitude = fromLon.text;
                            journeyModel.request.from = from;
                            var to = journeyModel.request.to;
                            to.name = "";
                            to.latitude = toLat.text;
                            to.longitude = toLon.text;
                            journeyModel.request.to = to;
                            parent.setupRequestCommon();
                        }
                    }
                    QQC2.Button {
                        text: "Clear"
                        onClicked: {
                            fromName.text = "";
                            fromLon.text = "";
                            fromLat.text = "";
                            toName.text = "";
                            toLon.text = "";
                            toLat.text = "";
                        }
                    }
                }

                RowLayout {
                    QQC2.ToolButton {
                        id: prevQueryButton
                        icon.name: "go-previous"
                        enabled: journeyModel.canQueryPrevious
                        onClicked: journeyModel.queryPrevious()
                    }
                    QQC2.ComboBox {
                        id: journeySelector
                        Layout.fillWidth: true
                        model: titleModel
                        textRole: "display"
                    }
                    QQC2.ToolButton {
                        id: nextQueryButton
                        icon.name: "go-next"
                        enabled: journeyModel.canQueryNext
                        onClicked: journeyModel.queryNext()
                    }
                }

                ListView {
                    Layout.fillHeight: true
                    Layout.fillWidth: true
                    model: journeyModel.data(journeyModel.index(journeySelector.currentIndex, 0), 256).sections
                    clip: true
                    delegate: journeyDelegate

                    QQC2.BusyIndicator {
                        anchors.centerIn: parent
                        running: journeyModel.loading
                    }

                    QQC2.Label {
                        anchors.centerIn: parent
                        width: parent.width
                        text: journeyModel.errorMessage
                        color: Kirigami.Theme.negativeTextColor
                        wrapMode: Text.Wrap
                    }
                }

            }
        }
    }
}
