/*
  Copyright (c) 2013-2020 Laurent Montel <montel@kde.org>

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "selectmulticollectionwidget.h"
#include "folderdialog/checkedcollectionwidget.h"

#include <CollectionFilterProxyModel>
#include <EntityTreeModel>

#include <KCheckableProxyModel>

#include <QVBoxLayout>
#include <QTreeView>

using namespace PimCommon;
SelectMultiCollectionWidget::SelectMultiCollectionWidget(const QString &mimetype, QWidget *parent)
    : QWidget(parent)
{
    initialize(mimetype);
}

SelectMultiCollectionWidget::SelectMultiCollectionWidget(const QString &mimetype, const QList<Akonadi::Collection::Id> &selectedCollection, QWidget *parent)
    : QWidget(parent)
    , mListCollection(selectedCollection)
{
    initialize(mimetype);
}

SelectMultiCollectionWidget::~SelectMultiCollectionWidget()
{
}

void SelectMultiCollectionWidget::initialize(const QString &mimetype)
{
    QVBoxLayout *vbox = new QVBoxLayout(this);
    vbox->setContentsMargins(0, 0, 0, 0);

    mCheckedCollectionWidget = new PimCommon::CheckedCollectionWidget(mimetype);

    connect(mCheckedCollectionWidget, &PimCommon::CheckedCollectionWidget::collectionAdded, this, &SelectMultiCollectionWidget::slotCollectionsTreeFetched);
    connect(mCheckedCollectionWidget, &PimCommon::CheckedCollectionWidget::collectionRemoved, this, &SelectMultiCollectionWidget::slotCollectionsTreeFetched);

    connect(mCheckedCollectionWidget->entityTreeModel(), &Akonadi::EntityTreeModel::collectionTreeFetched, this, &SelectMultiCollectionWidget::slotCollectionsTreeFetched);
    vbox->addWidget(mCheckedCollectionWidget);
}

void SelectMultiCollectionWidget::updateStatus(const QModelIndex &parent)
{
    const int nbCol = mCheckedCollectionWidget->checkableProxy()->rowCount(parent);
    for (int i = 0; i < nbCol; ++i) {
        const QModelIndex child = mCheckedCollectionWidget->checkableProxy()->index(i, 0, parent);

        const Akonadi::Collection col
            = mCheckedCollectionWidget->checkableProxy()->data(child, Akonadi::EntityTreeModel::CollectionRole).value<Akonadi::Collection>();

        if (mListCollection.contains(col.id())) {
            mCheckedCollectionWidget->checkableProxy()->setData(child, Qt::Checked, Qt::CheckStateRole);
        }
        updateStatus(child);
    }
}

void SelectMultiCollectionWidget::slotCollectionsTreeFetched()
{
    if (!mListCollection.isEmpty()) {
        updateStatus(QModelIndex());
    }
    mCheckedCollectionWidget->folderTreeView()->expandAll();
}

QVector<Akonadi::Collection> SelectMultiCollectionWidget::selectedCollection(const QModelIndex &parent) const
{
    QVector<Akonadi::Collection> lst;

    const int nbCol = mCheckedCollectionWidget->checkableProxy()->rowCount(parent);
    for (int i = 0; i < nbCol; ++i) {
        const QModelIndex child = mCheckedCollectionWidget->checkableProxy()->index(i, 0, parent);

        const Akonadi::Collection col
            = mCheckedCollectionWidget->checkableProxy()->data(child, Akonadi::EntityTreeModel::CollectionRole).value<Akonadi::Collection>();

        if (mCheckedCollectionWidget->checkableProxy()->data(child, Qt::CheckStateRole).value<int>()) {
            lst << col;
        }
        lst << selectedCollection(child);
    }
    return lst;
}
