//////////////////////////////////////////////////////////////////////////////
//
//    KEXIFUTILS.CPP
//
//    Copyright (C) 2002-2004 Renchi Raju <renchi at pooh.tam.uiuc.edu>
//                            Gilles CAULIER <caulier dot gilles at free.fr>
//                            Ralf Hoelzer <kde at ralfhoelzer.com>
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin Steet, Fifth Floor, Cambridge, MA 02110-1301, USA.
//
//////////////////////////////////////////////////////////////////////////////

#include <qstring.h>
#include <qfile.h>

#include <kdebug.h>

extern "C"
{
#include <sys/types.h>
#include <sys/stat.h>
#include <utime.h>
#include <unistd.h>
}

#include "kexifutils.h"
#include "kexifdata.h"

static void streamsync(QDataStream &s)
{
       s.device()->at(s.device()->at());
}

bool KExifUtils::writeOrientation(const QString& filename, KExifData::ImageOrientation orientation)
{
   QString str = "";
   return writeFile( filename, str, orientation);
}

bool KExifUtils::writeComment(const QString& filename, const QString& comment)
{
   return writeFile( filename, comment, KExifData::UNSPECIFIED );
}

bool KExifUtils::writeFile(const QString& filename, const QString& comment,
                           KExifData::ImageOrientation orientation)
{
    // position from start of exifsection
    unsigned int position = 0; 

    // stores single bytes read
    Q_UINT8 byte;

    /* Write to new Jpeg file and replace the EXIF data with the new data on the fly */
    QFile file( filename );
    if (!file.open( IO_ReadWrite ))
    {
        kdWarning() << "Failed to open file: "
                    << filename << endl;
        return false;
    }

    // save the current modification time of the file
    struct stat stbuf;
    ::stat(QFile::encodeName(filename), &stbuf);
    
    QDataStream stream( &file );

    // JPEG data is little endian
    stream.setByteOrder(QDataStream::LittleEndian);
 
    Q_UINT16 header;
    stream >> header;

    // check for valid JPEG header
    if(header != 0xd8ff)
    {
       kdDebug() << "No JPEG file." << endl;
       file.close();
       return false;
    }

    // skip until EXIF marker is found
    while(!stream.atEnd())
    {
      while(byte != 0xff) {
        stream >> byte;
      }
      stream >> byte;

      // consume 0xff's used for padding
      while(byte == 0xff)
        stream >> byte;

      // stop when we reach APP0 marker or start of image data
      if(byte == 0xe1 || byte == 0xc0)  // TODO: check more markers!
        break;
    }

    if(byte != 0xe1) {
      kdDebug() << "No EXIF information found." << endl;
      file.close();
      return false;
    }
    
    // get length of EXIF section
    Q_UINT16 sectionLen;
    stream >> sectionLen;

    // check for 'Exif' header
    Q_UINT8 exifHead[6];
    for( int i = 0; i < 6 ; i++ )
        stream >> exifHead[i];

    if( exifHead[0] != 0x45 || exifHead[1] != 0x78 || 
        exifHead[2] != 0x69 || exifHead[3] != 0x66 ||
        exifHead[4] != 0x00 || exifHead[5] != 0x00 ) {
      kdDebug() << "No valid EXIF header found." << endl;
      file.close();
      return false;
    }

    // get byte order of exif data
    Q_UINT16 byteOrder;
    stream >> byteOrder;

    if(byteOrder != 0x4949 && byteOrder != 0x4D4D)
    {
      kdDebug() << "EXIF byte order could not be determined." << endl;
      file.close();
      return false;
    } 

    // switch to Motorola byte order
    if(byteOrder == 0x4D4D)
    {
        stream.setByteOrder(QDataStream::BigEndian);
    } 

    // Check tag mark
    Q_UINT16 tagMark;
    stream >> tagMark;

    if( tagMark != 0x002A ) {
      kdDebug() << "could not read EXIF tag mark." << endl;
      file.close();
      return false;
    }

    // get offset of first IFD
    Q_UINT32 ifdOffset;
    stream >> ifdOffset;

    if( (Q_UINT16)ifdOffset > sectionLen - 2 || (Q_UINT16)ifdOffset < 2) {
      kdDebug() << "Invalid IFD offset in EXIF data." << endl;
      file.close();
      return false;
    }

    // We now read 8 bytes after start of Exif data
    position = 8;

    // seek forward to first IFD
    for(int i = 0 ; i < (Q_UINT16)ifdOffset - 8 ; i++ ) {
        stream >> byte;
        position++;
    }

    
    QMemArray<unsigned char> buf(sectionLen);

    if( !comment.isEmpty() ) 
    {
       // replace EXIF UserCommentTag
       Q_UINT32 userCommentOffset = 0;
       Q_UINT32 oldCommentLength = 0;

       int currentPosition = position;

       for(int i=0 ; i < (sectionLen - currentPosition) ; i++)
       {
          stream >> byte;
          buf[i] = byte;
          position++;

          // search for UserComment tag
          // this code is not perfect, but the probability that the sequence below is not
          // the UserComment tag is very small.
          if(i > 4 && ((buf[i]==0x00 && buf[i-1]==0x07 && buf[i-2]==0x92 && buf[i-3]==0x86)
                   || (buf[i]==0x07 && buf[i-1]==0x00 && buf[i-2]==0x86 && buf[i-3]==0x92)))
          {
             stream >> oldCommentLength;
             stream >> userCommentOffset;
             position+=8;
             kdDebug() << "Found UserComment offset: " << userCommentOffset << endl;
             kdDebug() << "Found oldCommentLength: " << oldCommentLength << endl;
             break;
          }
       }
       if(!userCommentOffset)
       {
          kdDebug() << "No EXIF UserComment found." << endl;
          file.close();
          return false;
       }

       while(position < userCommentOffset)
       {
          stream >> byte;
          position++;
       }

       streamsync(stream);
       // write character code ASCII into offset position
       stream << 0x49435341;
       stream << 0x00000049;

       // don't write more than the old field length
       unsigned int writeLen = (oldCommentLength < comment.length() + 8) ? oldCommentLength - 8 : comment.length(); 

       // write comment into stream
       for(unsigned int i = 0 ; i < writeLen ; i++)
       {
          stream << (Q_UINT8)comment.at(i).latin1();
       }

       // overwrite rest of old comment. Standard suggests \20, but that's not done in practice 
       for(unsigned int i=0 ; i < oldCommentLength - writeLen - 8; i++)
       {
          stream << (Q_UINT8)0x00;
       }
    } 
    else 
    {
       // replace EXIF orientation tag
        
       Q_UINT16 numberOfTags;
       stream >> numberOfTags;
       position += 2;
       kdDebug() << "Number of EXIF tags in IFD0 section:" << numberOfTags << endl;
       
       int currentPosition = position;

       for(int i=0 ; i < (sectionLen - currentPosition) ; i++)
       {
          stream >> byte;
          position++;
          buf[i] = byte;

          // search for Orientation tag
          // this code is not perfect, but the probability that the sequence below is not
          // the Orientation tag is very small.
          if( byteOrder == 0x4D4D ) {
              if(i > 8  && buf[i]==0x01 && buf[i-1]==0x00 && buf[i-2]==0x00 && buf[i-3]==0x00 
                      && buf[i-4]==0x03 && buf[i-5]==0x00 && buf[i-6]==0x12 && buf[i-7]==0x01)
              {
                 streamsync(stream);
                  stream << (Q_UINT8)0x00;
                  stream << (Q_UINT8)orientation;

                  kdDebug() << "Wrote Image Orientation: " << orientation << endl;
                  break;
              }
          } else {
              if(i > 8 && buf[i]==0x00 && buf[i-1]==0x00 && buf[i-2]==0x00 && buf[i-3]==0x01 
                     && buf[i-4]==0x00 && buf[i-5]==0x03 && buf[i-6]==0x01 && buf[i-7]==0x12)
              {
                 streamsync(stream);
                  stream << (Q_UINT8)orientation;
                  stream << (Q_UINT8)0x00;

                  kdDebug() << "Wrote Image Orientation: " << orientation << endl;
                  break;
              }
          }
       }
    }

    file.close();
    file.flush();

    // set the utime of the file back to original state
    struct utimbuf timbuf;
    timbuf.actime  = stbuf.st_atime;
    timbuf.modtime = stbuf.st_mtime;
    ::utime(QFile::encodeName(filename), &timbuf);
    
    kdDebug() << "Wrote file '" << filename.latin1() << "'." << endl;

    return true;
}

