#include <qlayout.h>
#include <qsplitter.h>
#include <qwidgetstack.h>
#include <qpixmap.h>
#include <qfile.h>
#include <qstringlist.h>
#include <qstrlist.h>
#include <qvaluelist.h>
#include <qcolor.h>
#include <qheader.h>
#include <qtimer.h>

#include <klocale.h>
#include <kdirlister.h>
#include <kurl.h>
#include <kimageio.h>
#include <kfiledialog.h>
#include <kmessagebox.h>

#include "albumfoldertree.h"
#include "camerafoldertree.h"
#include "albumiconview.h"
#include "cameraiconview.h"
#include "thumbnailsize.h"
#include "generalsettings.h"
#include "camerasettings.h"

#include "digikamview.h"
#include "digikamsplash.h"
#include "digikam.h"


DigikamView::DigikamView(QWidget *_parent)
    : QWidget(_parent)
{

    mCameraBusy = false;
    mAlbumBusy = false;
    mCameraLastMsg = "";
    mCameraLastVal = 0;
    mAlbumLastMsg = "";
    mAlbumLastVal = 0;

    mLayout = new QGridLayout(this);

    mSplitter = new QSplitter(this);

    mDirSplitter = new QSplitter(Qt::Vertical,mSplitter);
    mAlbumFolderTree = new AlbumFolderTree(mDirSplitter);
    mCameraFolderTree = new CameraFolderTree(mDirSplitter);

    mIconViewStack = new QWidgetStack(mSplitter);

    mAlbumIconView = new AlbumIconView(mIconViewStack);
    mCameraIconView = new CameraIconView(mIconViewStack);

    mIconViewStack->addWidget(mAlbumIconView, AlbumIconViewID);
    mIconViewStack->addWidget(mCameraIconView, CameraIconViewID);
    mIconViewStack->raiseWidget(AlbumIconViewID);

    mCameraFolderTree->setIconView(mCameraIconView);

    mSplitter->setOpaqueResize (true);
    mSplitter->setResizeMode( mDirSplitter, QSplitter::Stretch );
    mSplitter->setResizeMode( mIconViewStack, QSplitter::Stretch );

    mDirSplitter->setOpaqueResize(true);
    mDirSplitter->setResizeMode(mAlbumFolderTree, QSplitter::Stretch);
    mDirSplitter->setResizeMode(mCameraFolderTree, QSplitter::Stretch);

    QValueList<int> sizeList;
    sizeList.push_back(2);
    sizeList.push_back(5);
    mSplitter->setSizes (sizeList);

    QValueList<int> size2List;
    size2List.push_back(5);
    size2List.push_back(3);
    mDirSplitter->setSizes (size2List);

    mLayout->addWidget(mSplitter, 0, 0);


    mDirLister = new KDirLister();

    QStrList mimeStrList = QImageIO::inputFormats();
    QStringList mimeTypes;
    QStrListIterator it(mimeStrList);
    while (it.current() != 0) {
        QString mime("image/");
        mime += QString(*it).lower();
        mimeTypes.append(mime);
        ++it;
    }
    mDirLister->setMimeFilter(mimeTypes);


    setupConnections();

    mHighlightColor = QColor(255,128,0);
    mOrigColor = colorGroup().background();
    if (!mOrigColor.isValid())
        mOrigColor = QColor(255,255,255);
    highlightListView(mAlbumFolderTree);
}


DigikamView::~DigikamView()
{
    delete mDirLister;
}

void DigikamView::setupConnections()
{
    connect(mIconViewStack,SIGNAL(aboutToShow(int)),
            this,SLOT(slot_raisingIconView(int)));

    // -- DirLister Connections --------------------------------------

    connect(mDirLister,SIGNAL(newItems(const KFileItemList&)),
            this,SLOT(slot_dirListerNewItems(const KFileItemList&)) );

    connect(mDirLister,SIGNAL(deleteItem(KFileItem*)),
            this,SLOT(slot_dirListerDeleteItem(KFileItem*)) );

    connect(mDirLister,SIGNAL(clear()),
            this,SLOT(slot_dirListerClear()) );

    connect(mDirLister,SIGNAL(completed()),
            this,SLOT(slot_dirListerCompleted()) );

    connect(mDirLister,SIGNAL(refreshItems(const KFileItemList&)),
            this,SLOT(slot_dirListerRefreshItems(const KFileItemList&)));

    // -- AlbumFolderTree Connections ----------------------------------

    connect(mAlbumFolderTree, SIGNAL(signal_currentURLChanged(const KURL&)),
            this, SLOT(slot_newURLSelected(const KURL&)));

    connect(mAlbumFolderTree, SIGNAL(clicked(QListViewItem*)),
            this, SLOT(slot_albumFolderTreeClicked()));

    connect(mAlbumFolderTree, SIGNAL(signal_requestCameraDownload(const QStringList&,
                                                                  const QStringList&,
                                                                  const QString&)),
            this, SLOT(slot_cameraDownload(const QStringList&,
                                           const QStringList&,
                                           const QString&)));

    // -- AlbumView Connections --------------------------------------

    connect(mAlbumIconView, SIGNAL(signal_albumBusy(bool)),
            this, SLOT(slot_albumBusy(bool)));

    connect(mAlbumIconView, SIGNAL(signal_albumProgress(int)),
            this, SLOT(slot_albumProgress(int)));

    // -- CameraFolderTree Connections -------------------------------

    connect(mCameraFolderTree, SIGNAL(signal_cameraFolderClicked()),
            this, SLOT(slot_cameraFolderTreeClicked()));

    connect(mCameraFolderTree, SIGNAL(signal_cameraConnected(bool)),
            this, SIGNAL(signal_cameraConnected(bool)));

    connect(mCameraFolderTree, SIGNAL(signal_cameraStatusMsg(const QString&)),
            this, SLOT(slot_cameraStatusMsg(const QString&)));

    connect(mCameraFolderTree, SIGNAL(signal_cameraProgress(int)),
            this, SLOT(slot_cameraProgress(int)));

    connect(mCameraFolderTree, SIGNAL(signal_cameraBusy(bool)),
            this, SLOT(slot_cameraBusy(bool)));

    // -- CameraIconView Connections --------------------------------

    connect(mCameraIconView, SIGNAL(signal_downloadSelectedItems()),
            this, SLOT(slot_cameraDownload()));
    connect(mCameraIconView, SIGNAL(signal_deleteSelectedItems()),
            this, SLOT(slot_cameraDelete()));


}

void DigikamView::setAlbumPath(const QString& albumPath)
{
    if (KURL(albumPath).cmp(mAlbumPath,true))
        return;

    mAlbumPath = KURL(albumPath);
    mCurrentDownloadAlbum = KURL(albumPath);

    mAlbumFolderTree->setRootPath(mAlbumPath);
    mAlbumIconView->setPath(mAlbumPath);
    mDirLister->openURL(mAlbumPath);
}


void DigikamView::setThumbnailSize(const ThumbnailSize& thumbSize)
{
    if (mThumbnailSize == thumbSize) return;
    mThumbnailSize = thumbSize;

    if (mIconViewStack->visibleWidget() == mAlbumIconView) {
        if (mThumbnailSize != mAlbumIconView->getThumbnailSize())
            mAlbumIconView->resizeThumbnails(mThumbnailSize);
        return;
    }

    if (mIconViewStack->visibleWidget() == mCameraIconView) {
        if (mThumbnailSize !=  mCameraIconView->getThumbnailSize())
            mCameraIconView->resizeThumbnails(mThumbnailSize);
        return;
    }

}


QString DigikamView::getWhatsShowing() const
{
    QString albumName = mCurrentDownloadAlbum.path();

    albumName.remove(0,mAlbumPath.path().length());
    if (albumName.isEmpty())
        albumName = i18n("Main Album");

    if (mIconViewStack->visibleWidget() == mAlbumIconView) {
        QString msg(i18n("Showing Album: "));
        msg += albumName;
        return msg;
    }
    else if (mIconViewStack->visibleWidget() == mCameraIconView) {
        QString msg(i18n("Showing Camera Images"));
        msg += i18n(" (Downloads to Album ");
        msg += albumName;
	msg += ")";
        return msg;
    }

    return QString(i18n("Ready"));

}

// =======================================================================

void DigikamView::slot_dirListerNewItems(const KFileItemList& itemList)
{

    KFileItem *item;

    for (KFileItemListIterator it(itemList);
         (item = it.current()); ++it) {
        mAlbumIconView->addItem(item);
    }
}

void DigikamView::slot_dirListerDeleteItem(KFileItem* item)
{
    if (!item) return;
    mAlbumIconView->deleteItem(item);
}

void DigikamView::slot_dirListerClear()
{
    mAlbumIconView->clear();
}

void DigikamView::slot_dirListerCompleted()
{

}

void DigikamView::slot_dirListerRefreshItems(const
                                                  KFileItemList&
                                                  _itemList)
{
    KFileItemListIterator iterator(_itemList);
    KFileItem *item;

    while ((item = iterator.current()) != 0) {
        mAlbumIconView->refreshItem(item);
        ++iterator;
    }
}

// =======================================================================

void DigikamView::slot_newURLSelected(const KURL& url)
{
    if (url.isValid()) {
        mCurrentDownloadAlbum = url;
        mDirLister->stop();
        mAlbumIconView->setPath(url.path());
        mDirLister->openURL(url);
        if (mIconViewStack->visibleWidget() != mAlbumIconView)
            mIconViewStack->raiseWidget(AlbumIconViewID);
        emit signal_resetStatusBar();
        mAlbumIconView->resizeThumbnails(mThumbnailSize);
    }
}

// =======================================================================

void DigikamView::slot_albumFolderTreeClicked()
{
    if (mIconViewStack->visibleWidget() != mAlbumIconView) {
        mIconViewStack->raiseWidget(AlbumIconViewID);
        if (mThumbnailSize != mAlbumIconView->getThumbnailSize())
            mAlbumIconView->resizeThumbnails(mThumbnailSize);
        if (mAlbumBusy) {
            emit signal_statusMsg(mAlbumLastMsg);
            emit signal_progress(mAlbumLastVal);
        }
        else {
            emit signal_resetStatusBar();
        }
    }
}

void DigikamView::slot_cameraFolderTreeClicked()
{
    if (mIconViewStack->visibleWidget() != mCameraIconView) {
        mIconViewStack->raiseWidget(CameraIconViewID);
        if (mThumbnailSize != mCameraIconView->getThumbnailSize())
            mCameraIconView->resizeThumbnails(mThumbnailSize);
        if (mCameraBusy) {
            emit signal_statusMsg(mCameraLastMsg);
            emit signal_progress(mCameraLastVal);
        }
        else {
            emit signal_resetStatusBar();
        }
    }
}


// =======================================================================

void DigikamView::slot_cameraStatusMsg(const QString& msg)
{
    mCameraLastMsg = msg;
    if (mIconViewStack->visibleWidget() != mAlbumIconView)
        emit signal_statusMsg(msg);
}

void DigikamView::slot_cameraProgress(int val)
{
    mCameraLastVal = val;
    if (mIconViewStack->visibleWidget() != mAlbumIconView)
        emit signal_progress(val);
}

void DigikamView::slot_cameraBusy(bool busy)
{
    if (busy) {
        if (!mCameraBusy) {
            mCameraBusy = true;
            if (!mAlbumBusy) emit signal_cameraBusy();
        }
    }
    else {
        if (mCameraBusy) {
            mCameraBusy = false;
            if (!mAlbumBusy) emit signal_cameraIdle();
        }
    }
}

// =======================================================================

void DigikamView::slot_albumBusy(bool busy)
{
    mAlbumBusy = busy;
    if (!busy && mIconViewStack->visibleWidget() == mAlbumIconView)
        emit signal_resetStatusBar();

}

void DigikamView::slot_albumProgress(int val)
{
    mAlbumLastMsg = i18n("Generating Thumbnails");
    mAlbumLastVal = val;
    if (mIconViewStack->visibleWidget() == mAlbumIconView) {
        emit signal_statusMsg(mAlbumLastMsg);
        emit signal_progress(mAlbumLastVal);
    }
}


// =======================================================================

void DigikamView::slot_hideFolders()
{
    if (mDirSplitter->isHidden())
        mDirSplitter->show();
    else
        mDirSplitter->hide();
}

void DigikamView::slot_hideCameraFolders()
{
    if (mCameraFolderTree->isHidden())
        mCameraFolderTree->show();
    else
        mCameraFolderTree->hide();
}

void DigikamView::slot_hideAlbumFolders()
{
    if (mAlbumFolderTree->isHidden())
        mAlbumFolderTree->show();
    else
        mAlbumFolderTree->hide();
}

void DigikamView::slot_selectAll()
{
    if (mIconViewStack->visibleWidget() == mAlbumIconView) {
        mAlbumIconView->selectAll(true);
        return;
    }

    if (mIconViewStack->visibleWidget() == mCameraIconView) {
        mCameraIconView->selectAll(true);
        return;
    }

}

void DigikamView::slot_selectNone()
{
    if (mIconViewStack->visibleWidget() == mAlbumIconView) {
        mAlbumIconView->clearSelection();
        return;
    }

    if (mIconViewStack->visibleWidget() == mCameraIconView) {
        mCameraIconView->clearSelection();
        return;
    }

}

void DigikamView::slot_selectInvert()
{
    if (mIconViewStack->visibleWidget() == mAlbumIconView) {
        mAlbumIconView->invertSelection();
        return;
    }

    if (mIconViewStack->visibleWidget() == mCameraIconView) {
        mCameraIconView->invertSelection();
        return;
    }

}

// =======================================================================

void DigikamView::slot_cameraConnect()
{
    mCameraIconView->setThumbnailSize(mThumbnailSize);

    mCameraFolderTree->camera_connect();
    mIconViewStack->raiseWidget(CameraIconViewID);
}

void DigikamView::slot_cameraDisconnect()
{
    mCameraFolderTree->camera_disconnect();
}

void DigikamView::slot_cameraCancel()
{
    mCameraFolderTree->camera_cancel();
}

void DigikamView::slot_cameraDownload()
{
    if (mIconViewStack->visibleWidget() == mAlbumIconView)
        return;
    if (mIconViewStack->visibleWidget() == mCameraIconView)
        mCameraFolderTree->camera_download(mCurrentDownloadAlbum.path());
}

void DigikamView::slot_cameraDownload(const QStringList& nameList,
                                      const QStringList& folderList,
                                      const QString& downloadDir)
{
    mCameraFolderTree->camera_download(nameList, folderList, downloadDir);
}

void DigikamView::slot_cameraDownloadAll()
{
    mCameraFolderTree->camera_downloadAll(mCurrentDownloadAlbum.path());
}

void DigikamView::slot_cameraDelete()
{
    if (mIconViewStack->visibleWidget() == mAlbumIconView)
        return;
    if (mIconViewStack->visibleWidget() == mCameraIconView)
        mCameraFolderTree->camera_delete();
}

void DigikamView::slot_cameraDeleteAll()
{
    mCameraFolderTree->camera_deleteAll();
}

void DigikamView::slot_cameraUpload()
{
    QString reason;
    if (!(mCameraFolderTree->cameraReadyForUpload(reason))) {
        KMessageBox::error(0,reason);
        return;
    }

    KURL::List fileList = KFileDialog::getOpenURLs(QString::null,
                                                   QString::null,this);
    if (!fileList.isEmpty()) {
        KURL::List::iterator it;
        for (it = fileList.begin(); it != fileList.end(); ++it) {
            mCameraFolderTree->camera_upload((*it).path());
        }
    }

}

void DigikamView::slot_cameraInfo()
{
    mCameraFolderTree->camera_information();
}

// =======================================================================

void DigikamView::slot_createNewAlbum()
{
    mAlbumFolderTree->makeDir();
}

void DigikamView::slot_deleteAlbum()
{
    mAlbumFolderTree->rmDir();
}
void DigikamView::slot_albumRegenerateThumbnails()
{
    mAlbumIconView->regenerateThumbnails();
}

void DigikamView::slot_albumSlideShow()
{
    mAlbumIconView->startSlideShow();
}

void DigikamView::slot_albumPauseThumbnails()
{

    mAlbumIconView->pauseThumbnailGeneration();
}

void DigikamView::slot_albumRefresh()
{
    KURL url(mDirLister->url());

    if (url.isValid()) {
        mDirLister->stop();
        mAlbumIconView->setPath(url);
        mDirLister->openURL(url,false,true);
    }

}

// =======================================================================

void DigikamView::slot_raisingIconView(int id)
{

    switch(id) {
    case(AlbumIconViewID): {
        highlightListView(mAlbumFolderTree);
        unhighlightListView(mCameraFolderTree);
        break;
    }
    default: {
        highlightListView(mCameraFolderTree);
        unhighlightListView(mAlbumFolderTree);
        break;
    }
    }
}

void DigikamView::highlightListView(KListView* view)
{
    view->header()->setPaletteBackgroundColor(mHighlightColor);
}

void DigikamView::unhighlightListView(KListView* view)
{
    view->header()->setPaletteBackgroundColor(mOrigColor);
}

void DigikamView::applyGeneralSettings(const GeneralSettings& genSettings)
{
    setAlbumPath(genSettings.getAlbumPath());
    setThumbnailSize(ThumbnailSize(genSettings.getDefaultIconSize()));

    mCameraIconView->setBackGroundColor(
        genSettings.getCameraViewBackgroundColor());

    mAlbumIconView->setSlideShowParams(
        genSettings.getSlideShowDelay(),
        genSettings.getSlideShowTransitionEffect(),
        genSettings.getSlideShowFileName());

    mAlbumIconView->setImageViewerParams(
        genSettings.getUseInternalImageViewer(),
        genSettings.getImageViewer());

    mCameraIconView->setImageViewerParams(
        genSettings.getUseInternalImageViewer(),
        genSettings.getImageViewer());
}

void DigikamView::applyCameraSettings(const CameraSettings& camSettings)
{
    mCameraFolderTree->camera_change(camSettings.getCameraModel(),
                                     camSettings.getCameraPort(),
                                     camSettings.getCameraPath());
}

