/***************************************************************************
 *   Copyright (C) 2006 by Till Adam <adam@kde.org>                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Library General Public     *
 *   License along with this program; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.             *
 ***************************************************************************/
#ifndef AKONADIHANDLER_H
#define AKONADIHANDLER_H

#include <QObject>

#include "global.h"
#include "exception.h"
#include "connection.h"

#include <private/protocol_p.h>

namespace Akonadi
{

namespace Server
{

class Response;

AKONADI_EXCEPTION_MAKE_INSTANCE(HandlerException);

/**
  \defgroup akonadi_server_handler Command handlers

  All commands supported by the Akonadi server are implemented as sub-classes of Akonadi::Handler.
*/

/**
The handler interfaces describes an entity capable of handling an AkonadiIMAP command.*/
class Handler
{
public:
    Handler() = default;
    virtual ~Handler() = default;

    /**
     * Set the tag of the command to be processed, and thus of the response
     * generated by this handler.
     * @param tag The command tag, an alphanumerical string, normally.
     */
    void setTag(quint64 tag);

    /**
     * The tag of the command associated with this handler.
     */
    quint64 tag() const;

    void setCommand(const Protocol::CommandPtr &cmd);
    Protocol::CommandPtr command() const;

    /**
     * Find a handler for a command that is always allowed, like LOGOUT.
     * @param cmd the command string
     * @return an instance to the handler. The handler is deleted after @see handelLine is executed. The caller needs to delete the handler in case an exception is thrown from handelLine.
     */
    static Handler *findHandlerForCommandAlwaysAllowed(Protocol::Command::Type cmd);

    /**
     * Find a handler for a command that is allowed when the client is not yet authenticated, like LOGIN.
     * @param cmd the command string
     * @return an instance to the handler. The handler is deleted after @see handelLine is executed. The caller needs to delete the handler in case an exception is thrown from handelLine.
     */
    static Handler *findHandlerForCommandNonAuthenticated(Protocol::Command::Type cmd);

    /**
     * Find a handler for a command that is allowed when the client is authenticated, like LIST, FETCH, etc.
     * @param cmd the command string
     * @return an instance to the handler. The handler is deleted after @see handelLine is executed. The caller needs to delete the handler in case an exception is thrown from handelLine.
     */
    static Handler *findHandlerForCommandAuthenticated(Protocol::Command::Type cmd);

    void setConnection(Connection *connection);
    Connection *connection() const;
    DataStore *storageBackend() const;

    bool failureResponse(const char *response);
    bool failureResponse(const QByteArray &response);
    bool failureResponse(const QString &response);

    template<typename T>
    inline bool successResponse();
    template<typename T>
    inline bool successResponse(T &&response);

    template<typename T>
    inline void sendResponse(T &&response);
    template<typename T>
    inline void sendResponse();

    /**
     * Parse and handle the IMAP message using the streaming parser. The implementation MUST leave the trailing newline character(s) in the stream!
     * @return true if parsed successfully, false in case of parse failure
     */
    virtual bool parseStream() = 0;

    bool checkScopeConstraints(const Scope &scope, int permittedScopes);

private:
    quint64 m_tag = 0;
    Connection *m_connection = nullptr;
    bool m_sentFailureResponse = false;

protected:
    Protocol::CommandPtr m_command;
};

template<typename T>
inline bool Handler::successResponse()
{
    sendResponse<T>(T{});
    return true;
}

template<typename T>
inline bool Handler::successResponse(T &&response)
{
    sendResponse<T>(std::move(response));
    return true;
}

template<typename T>
inline void Handler::sendResponse()
{
    m_connection->sendResponse<T>(T{});
}

template<typename T>
inline void Handler::sendResponse(T &&response)
{
    m_connection->sendResponse<T>(std::move(response));
}

} // namespace Server
} // namespace Akonadi

#endif
