//
// Description: Kate Plugin for GDB integration
//
//
// Copyright (c) 2010 Ian Wakeling <ian.wakeling@ntlworld.com>
// Copyright (c) 2010-2014 Kåre Särs <kare.sars@iki.fi>
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Library General Public
//  License version 2 as published by the Free Software Foundation.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Library General Public License for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; see the file COPYING.LIB.  If not, write to
//  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
//  Boston, MA 02110-1301, USA.

#include "plugin_kategdb.h"

#include <QFile>
#include <QFileInfo>
#include <QTabWidget>
#include <QToolBar>
#include <QSplitter>
#include <QLayout>
#include <QTextEdit>
#include <QScrollBar>
#include <QTreeWidget>
#include <QKeyEvent>
#include <QFontDatabase>

#include <QAction>
#include <KXMLGUIFactory>
#include <KActionCollection>
#include <KConfigGroup>
#include <QMenu>

#include <klocalizedstring.h>
#include <kcolorscheme.h>
#include <kpluginfactory.h>
#include <kpluginloader.h>
#include <kaboutdata.h>
#include <khistorycombobox.h>

#include <ktexteditor/view.h>
#include <ktexteditor/document.h>
#include <ktexteditor/markinterface.h>
#include <ktexteditor/editor.h>


K_PLUGIN_FACTORY_WITH_JSON (KatePluginGDBFactory, "kategdbplugin.json", registerPlugin<KatePluginGDB>();)

KatePluginGDB::KatePluginGDB(QObject* parent, const VariantList&)
:   KTextEditor::Plugin(parent)
{
    // FIXME KF5 KGlobal::locale()->insertCatalog("kategdbplugin");
}

KatePluginGDB::~KatePluginGDB()
{
}

QObject* KatePluginGDB::createView(KTextEditor::MainWindow* mainWindow)
{
    return new KatePluginGDBView(this, mainWindow);
}

KatePluginGDBView::KatePluginGDBView(KTextEditor::Plugin *plugin, KTextEditor::MainWindow *mainWin)
:   QObject(mainWin), m_mainWin(mainWin)
{
    m_lastExecUrl = QUrl();
    m_lastExecLine = -1;
    m_lastExecFrame = 0;
    m_kateApplication = KTextEditor::Editor::instance()->application();
    m_focusOnInput = true;
    m_activeThread = -1;

    KXMLGUIClient::setComponentName (QLatin1String("kategdb"), i18n ("Kate GDB"));
    setXMLFile(QLatin1String("ui.rc"));

    m_toolView = m_mainWin->createToolView(plugin, i18n("Debug View"),
                                           KTextEditor::MainWindow::Bottom,
                                           QIcon(QStringLiteral(":/kategdb/22-actions-debug-kategdb.png")),
                                           i18n("Debug View"));

    m_localsStackToolView = m_mainWin->createToolView(plugin, i18n("Locals and Stack"),
                                                      KTextEditor::MainWindow::Right,
                                                      QIcon(QStringLiteral(":/kategdb/22-actions-debug-kategdb.png")),
                                                      i18n("Locals and Stack"));

    m_tabWidget = new QTabWidget(m_toolView);
    // Output
    m_outputArea = new QTextEdit();
    m_outputArea->setAcceptRichText(false );
    m_outputArea->setReadOnly(true);
    m_outputArea->setUndoRedoEnabled(false);
    // fixed wide font, like konsole
    m_outputArea->setFont(QFontDatabase::systemFont(QFontDatabase::FixedFont));
    // alternate color scheme, like konsole
    KColorScheme schemeView(QPalette::Active, KColorScheme::View);
    m_outputArea->setTextBackgroundColor(schemeView.foreground().color());
    m_outputArea->setTextColor(schemeView.background().color());
    QPalette p = m_outputArea->palette ();
    p.setColor(QPalette::Base, schemeView.foreground().color());
    m_outputArea->setPalette(p);

    // input
    m_inputArea = new KHistoryComboBox(true);
    connect(m_inputArea,  SIGNAL(returnPressed()), this, SLOT(slotSendCommand()));
    QHBoxLayout *inputLayout = new QHBoxLayout();
    inputLayout->addWidget(m_inputArea, 10);
    inputLayout->setContentsMargins(0,0,0,0);
    m_outputArea->setFocusProxy(m_inputArea); // take the focus from the outputArea

    m_gdbPage = new QWidget();
    QVBoxLayout *layout = new QVBoxLayout(m_gdbPage);
    layout->addWidget(m_outputArea);
    layout->addLayout(inputLayout);
    layout->setStretch(0, 10);
    layout->setContentsMargins(0,0,0,0);
    layout->setSpacing(0);

    // stack page
    QWidget *stackContainer = new QWidget();
    QVBoxLayout *stackLayout = new QVBoxLayout(stackContainer);
    m_threadCombo = new QComboBox();
    m_stackTree = new QTreeWidget();
    stackLayout->addWidget(m_threadCombo);
    stackLayout->addWidget(m_stackTree);
    stackLayout->setStretch(0, 10);
    stackLayout->setContentsMargins(0,0,0,0);
    stackLayout->setSpacing(0);
    QStringList headers;
    headers << QStringLiteral("  ") << i18nc("Column label (frame number)", "Nr") << i18nc("Column label", "Frame");
    m_stackTree->setHeaderLabels(headers);
    m_stackTree->setRootIsDecorated(false);
    m_stackTree->resizeColumnToContents(0);
    m_stackTree->resizeColumnToContents(1);
    m_stackTree->setAutoScroll(false);
    connect(m_stackTree, SIGNAL(itemActivated(QTreeWidgetItem*,int)),
             this, SLOT(stackFrameSelected()));

    connect(m_threadCombo, SIGNAL(currentIndexChanged(int)),
             this, SLOT(threadSelected(int)));


    m_localsView = new LocalsView();

    QSplitter *locStackSplitter = new QSplitter(m_localsStackToolView);
    locStackSplitter->addWidget(m_localsView);
    locStackSplitter->addWidget(stackContainer);
    locStackSplitter->setOrientation(Qt::Vertical);

    // config page
    m_configView = new ConfigView(NULL, mainWin);

    m_ioView = new IOView();
    connect(m_configView, SIGNAL(showIO(bool)),
             this,       SLOT(showIO(bool)));

    m_tabWidget->addTab(m_gdbPage, i18nc("Tab label", "GDB Output"));
    m_tabWidget->addTab(m_configView, i18nc("Tab label", "Settings"));

    m_debugView  = new DebugView(this);
    connect(m_debugView, SIGNAL(readyForInput(bool)),
             this,        SLOT(enableDebugActions(bool)));

    connect(m_debugView, SIGNAL(outputText(QString)),
             this,        SLOT(addOutputText(QString)));

    connect(m_debugView, SIGNAL(outputError(QString)),
             this,        SLOT(addErrorText(QString)));

    connect(m_debugView, SIGNAL(debugLocationChanged(QUrl,int)),
             this,        SLOT(slotGoTo(QUrl,int)));

    connect(m_debugView, SIGNAL(breakPointSet(QUrl,int)),
             this,        SLOT(slotBreakpointSet(QUrl,int)));

    connect(m_debugView, SIGNAL(breakPointCleared(QUrl,int)),
             this,        SLOT(slotBreakpointCleared(QUrl,int)));

    connect(m_debugView, SIGNAL(clearBreakpointMarks()),
             this,        SLOT(clearMarks()));

    connect(m_debugView, SIGNAL(programEnded()),
             this,        SLOT(programEnded()));

    connect(m_debugView, SIGNAL(gdbEnded()),
             this,        SLOT(programEnded()));

    connect(m_debugView, SIGNAL(gdbEnded()),
             this,        SLOT(gdbEnded()));

    connect(m_debugView, SIGNAL(stackFrameInfo(QString,QString)),
             this,        SLOT(insertStackFrame(QString,QString)));

    connect(m_debugView, SIGNAL(stackFrameChanged(int)),
             this,        SLOT(stackFrameChanged(int)));

    connect(m_debugView,  SIGNAL(infoLocal(QString)),
             m_localsView, SLOT(addLocal(QString)));

    connect(m_debugView, SIGNAL(threadInfo(int,bool)),
             this,        SLOT(insertThread(int,bool)));

    connect(m_localsView, SIGNAL(localsVisible(bool)),
             m_debugView,  SLOT(slotQueryLocals(bool)));

    // Actions
    m_configView->registerActions(actionCollection());

    QAction* a = actionCollection()->addAction(QStringLiteral("debug"));
    a->setText(i18n("Start Debugging"));
    a->setIcon(QIcon(QStringLiteral(":/kategdb/22-actions-debug-kategdb.png")));
    connect(   a,      SIGNAL(triggered(bool)),
                this,   SLOT(slotDebug()));

    a = actionCollection()->addAction(QStringLiteral("kill"));
    a->setText(i18n("Kill / Stop Debugging"));
    a->setIcon(QIcon::fromTheme(QStringLiteral("media-playback-stop")));
    connect(   a,         SIGNAL(triggered(bool)),
                m_debugView, SLOT(slotKill()));

    a = actionCollection()->addAction(QStringLiteral("rerun"));
    a->setText(i18n("Restart Debugging"));
    a->setIcon(QIcon::fromTheme(QStringLiteral("view-refresh")));
    connect(   a,         SIGNAL(triggered(bool)),
                this, SLOT(slotRestart()));

    a = actionCollection()->addAction(QStringLiteral("toggle_breakpoint"));
    a->setText(i18n("Toggle Breakpoint / Break"));
    a->setIcon(QIcon::fromTheme(QStringLiteral("media-playback-pause")));
    connect(   a,      SIGNAL(triggered(bool)),
                this,   SLOT(slotToggleBreakpoint()));

    a = actionCollection()->addAction(QStringLiteral("step_in"));
    a->setText(i18n("Step In"));
    a->setIcon(QIcon::fromTheme(QStringLiteral("debug-step-into")));
    connect(   a,      SIGNAL(triggered(bool)),
                m_debugView,   SLOT(slotStepInto()));

    a = actionCollection()->addAction(QStringLiteral("step_over"));
    a->setText(i18n("Step Over"));
    a->setIcon(QIcon::fromTheme(QStringLiteral("debug-step-over")));
    connect(   a,      SIGNAL(triggered(bool)),
                m_debugView,   SLOT(slotStepOver()));

    a = actionCollection()->addAction(QStringLiteral("step_out"));
    a->setText(i18n("Step Out"));
    a->setIcon(QIcon::fromTheme(QStringLiteral("debug-step-out")));
    connect(   a,      SIGNAL(triggered(bool)),
                m_debugView, SLOT(slotStepOut()));

    a = actionCollection()->addAction(QStringLiteral("move_pc"));
    a->setText(i18nc("Move Program Counter (next execution)", "Move PC"));
    connect(   a,      SIGNAL(triggered(bool)),
                this,   SLOT(slotMovePC()));

    a = actionCollection()->addAction(QStringLiteral("run_to_cursor"));
    a->setText(i18n("Run To Cursor"));
    a->setIcon(QIcon::fromTheme(QStringLiteral("debug-run-cursor")));
    connect(   a,      SIGNAL(triggered(bool)),
                this,   SLOT(slotRunToCursor()));

    a = actionCollection()->addAction(QStringLiteral("continue"));
    a->setText(i18n("Continue"));
    a->setIcon(QIcon::fromTheme(QStringLiteral("media-playback-start")));
    connect(   a,      SIGNAL(triggered(bool)),
                m_debugView, SLOT(slotContinue()));

    a = actionCollection()->addAction(QStringLiteral("print_value"));
    a->setText(i18n("Print Value"));
    a->setIcon(QIcon::fromTheme(QStringLiteral("document-preview")));
    connect(   a,      SIGNAL(triggered(bool)),
                this, SLOT(slotValue()));

    // popup context m_menu
    m_menu = new KActionMenu(i18n("Debug"), this);
    actionCollection()->addAction(QStringLiteral("popup_gdb"), m_menu);
    connect(m_menu->menu(), SIGNAL(aboutToShow()), this, SLOT(aboutToShowMenu()));

    m_breakpoint = m_menu->menu()->addAction(i18n("popup_breakpoint"),
                                         this, SLOT(slotToggleBreakpoint()));

    QAction* popupAction = m_menu->menu()->addAction(i18n("popup_run_to_cursor"),
                                                   this, SLOT(slotRunToCursor()));
    popupAction->setText(i18n("Run To Cursor"));
    popupAction = m_menu->menu()->addAction(QStringLiteral("move_pc"),
                                          this, SLOT(slotMovePC()));
    popupAction->setText(i18nc("Move Program Counter (next execution)", "Move PC"));

    enableDebugActions(false);

    connect(m_mainWin, SIGNAL(unhandledShortcutOverride(QEvent*)),
            this, SLOT(handleEsc(QEvent*)));

    m_toolView->installEventFilter(this);

    m_mainWin->guiFactory()->addClient(this);
}

KatePluginGDBView::~KatePluginGDBView()
{
    m_mainWin->guiFactory()->removeClient(this);
    delete m_toolView;
    delete m_localsStackToolView;
}

void KatePluginGDBView::readSessionConfig( const KConfigGroup& config)
{
    m_configView->readConfig(config);
}

void KatePluginGDBView::writeSessionConfig(KConfigGroup& config)
{
    m_configView->writeConfig(config);
}

void KatePluginGDBView::slotDebug()
{
    disconnect(m_ioView, SIGNAL(stdOutText(QString)), 0, 0);
    disconnect(m_ioView, SIGNAL(stdErrText(QString)), 0, 0);
    if (m_configView->showIOTab()) {
        connect(m_ioView, SIGNAL(stdOutText(QString)), m_ioView, SLOT(addStdOutText(QString)));
        connect(m_ioView, SIGNAL(stdErrText(QString)), m_ioView, SLOT(addStdErrText(QString)));
    }
    else {
        connect(m_ioView, SIGNAL(stdOutText(QString)), this, SLOT(addOutputText(QString)));
        connect(m_ioView, SIGNAL(stdErrText(QString)), this, SLOT(addErrorText(QString)));
    }
    QStringList ioFifos;
    ioFifos << m_ioView->stdinFifo();
    ioFifos << m_ioView->stdoutFifo();
    ioFifos << m_ioView->stderrFifo();

    enableDebugActions(true);
    m_mainWin->showToolView(m_toolView);
    m_tabWidget->setCurrentWidget(m_gdbPage);
    QScrollBar *sb = m_outputArea->verticalScrollBar();
    sb->setValue(sb->maximum());
    m_localsView->clear();

    m_debugView->runDebugger(m_configView->currentTarget(), ioFifos);
}

void KatePluginGDBView::slotRestart()
{
    m_mainWin->showToolView(m_toolView);
    m_tabWidget->setCurrentWidget(m_gdbPage);
    QScrollBar *sb = m_outputArea->verticalScrollBar();
    sb->setValue(sb->maximum());
    m_localsView->clear();

    m_debugView->slotReRun();
}

void KatePluginGDBView::aboutToShowMenu()
{
    if (!m_debugView->debuggerRunning() || m_debugView->debuggerBusy()) {
        m_breakpoint->setText(i18n("Insert breakpoint"));
        m_breakpoint->setDisabled(true);
        return;
    }

    m_breakpoint->setDisabled(false);

    KTextEditor::View* editView = m_mainWin->activeView();
    QUrl               url = editView->document()->url();
    int                line = editView->cursorPosition().line();

    line++; // GDB uses 1 based line numbers, kate uses 0 based...

    if (m_debugView->hasBreakpoint(url, line)) {
        m_breakpoint->setText(i18n("Remove breakpoint"));
    }
    else {
        m_breakpoint->setText(i18n("Insert breakpoint"));
    }
}

void KatePluginGDBView::slotToggleBreakpoint()
{
    if (!actionCollection()->action(QStringLiteral("continue"))->isEnabled()) {
        m_debugView->slotInterrupt();
    }
    else {
        KTextEditor::View* editView = m_mainWin->activeView();
        QUrl               currURL  = editView->document()->url();
        int                line     = editView->cursorPosition().line();

        m_debugView->toggleBreakpoint(currURL, line + 1);
    }
}

void KatePluginGDBView::slotBreakpointSet(const QUrl &file, int line)
{
    KTextEditor::MarkInterface* iface =
    qobject_cast<KTextEditor::MarkInterface*>(m_kateApplication->findUrl(file));

    if (iface) {
        iface->setMarkDescription(KTextEditor::MarkInterface::BreakpointActive, i18n("Breakpoint"));
        iface->setMarkPixmap(KTextEditor::MarkInterface::BreakpointActive,
                             QIcon::fromTheme(QStringLiteral("media-playback-pause")).pixmap(10,10));
        iface->addMark(line, KTextEditor::MarkInterface::BreakpointActive);
    }
}

void KatePluginGDBView::slotBreakpointCleared(const QUrl &file, int line)
{
    KTextEditor::MarkInterface* iface =
    qobject_cast<KTextEditor::MarkInterface*>(m_kateApplication->findUrl(file));

    if (iface) {
        iface->removeMark(line, KTextEditor::MarkInterface::BreakpointActive);
    }
}

void KatePluginGDBView::slotMovePC()
{
    KTextEditor::View*  editView = m_mainWin->activeView();
    QUrl                currURL = editView->document()->url();
    KTextEditor::Cursor cursor = editView->cursorPosition();

    m_debugView->movePC(currURL, cursor.line() + 1);
}

void KatePluginGDBView::slotRunToCursor()
{
    KTextEditor::View*  editView = m_mainWin->activeView();
    QUrl                currURL = editView->document()->url();
    KTextEditor::Cursor cursor = editView->cursorPosition();

    // GDB starts lines from 1, kate returns lines starting from 0 (displaying 1)
    m_debugView->runToCursor(currURL, cursor.line() + 1);
}

void KatePluginGDBView::slotGoTo(const QUrl &url, int lineNum)
{
    // skip not existing files
    if (!QFile::exists (url.toLocalFile ())) {
        m_lastExecLine = -1;
        return;
    }

    m_lastExecUrl = url;
    m_lastExecLine = lineNum;

    KTextEditor::View*  editView = m_mainWin->openUrl(m_lastExecUrl);
    editView->setCursorPosition(KTextEditor::Cursor(m_lastExecLine, 0));
    m_mainWin->window()->raise();
    m_mainWin->window()->setFocus();
}

void KatePluginGDBView::enableDebugActions(bool enable)
{
    actionCollection()->action(QStringLiteral("step_in"      ))->setEnabled(enable);
    actionCollection()->action(QStringLiteral("step_over"    ))->setEnabled(enable);
    actionCollection()->action(QStringLiteral("step_out"     ))->setEnabled(enable);
    actionCollection()->action(QStringLiteral("move_pc"      ))->setEnabled(enable);
    actionCollection()->action(QStringLiteral("run_to_cursor"))->setEnabled(enable);
    actionCollection()->action(QStringLiteral("popup_gdb"    ))->setEnabled(enable);
    actionCollection()->action(QStringLiteral("continue"     ))->setEnabled(enable);
    actionCollection()->action(QStringLiteral("print_value"  ))->setEnabled(enable);

    // "toggle breakpoint" doubles as interrupt while the program is running
    actionCollection()->action(QStringLiteral("toggle_breakpoint"))->setEnabled(m_debugView->debuggerRunning());
    actionCollection()->action(QStringLiteral("kill"             ))->setEnabled(m_debugView->debuggerRunning());
    actionCollection()->action(QStringLiteral("rerun"            ))->setEnabled(m_debugView->debuggerRunning());

    m_inputArea->setEnabled(enable);
    m_threadCombo->setEnabled(enable);
    m_stackTree->setEnabled(enable);
    m_localsView->setEnabled(enable);

    if (enable)  {
        m_inputArea->setFocusPolicy(Qt::WheelFocus);

        if (m_focusOnInput || m_configView->takeFocusAlways()) {
            m_inputArea->setFocus();
            m_focusOnInput = false;
        }
        else {
            m_mainWin->activeView()->setFocus();
        }
    }
    else {
        m_inputArea->setFocusPolicy(Qt::NoFocus);
        if (m_mainWin->activeView()) m_mainWin->activeView()->setFocus();
    }

    m_ioView->enableInput(!enable && m_debugView->debuggerRunning());

    if ((m_lastExecLine > -1))
    {
        KTextEditor::MarkInterface* iface =
        qobject_cast<KTextEditor::MarkInterface*>(m_kateApplication->findUrl(m_lastExecUrl));

        if (iface) {
            if (enable) {
                iface->setMarkDescription(KTextEditor::MarkInterface::Execution, i18n("Execution point"));
                iface->setMarkPixmap(KTextEditor::MarkInterface::Execution, QIcon::fromTheme(QStringLiteral("arrow-right")).pixmap(10,10));
                iface->addMark(m_lastExecLine, KTextEditor::MarkInterface::Execution);
            }
            else {
                iface->removeMark(m_lastExecLine, KTextEditor::MarkInterface::Execution);
            }
        }
    }
}

void KatePluginGDBView::programEnded()
{
    // don't set the execution mark on exit
    m_lastExecLine = -1;
    m_stackTree->clear();
    m_localsView->clear();
    m_threadCombo->clear();

    // Indicate the state change by showing the debug outputArea
    m_mainWin->showToolView(m_toolView);
    m_tabWidget->setCurrentWidget(m_gdbPage);
}

void KatePluginGDBView::gdbEnded()
{
    m_outputArea->clear();
    m_localsView->clear();
    m_ioView->clearOutput();
    clearMarks();
}

void KatePluginGDBView::clearMarks()
{
    KTextEditor::MarkInterface* iface;
    foreach (KTextEditor::Document* doc, m_kateApplication->documents()) {
        iface = qobject_cast<KTextEditor::MarkInterface*>(doc);
        if (iface) {
            const QHash<int, KTextEditor::Mark*> marks = iface->marks();
            QHashIterator<int, KTextEditor::Mark*> i(marks);
            while (i.hasNext()) {
                i.next();
                if ((i.value()->type == KTextEditor::MarkInterface::Execution) ||
                    (i.value()->type == KTextEditor::MarkInterface::BreakpointActive)) {
                    iface->removeMark(i.value()->line, i.value()->type);
                    }
            }
        }
    }
}

void KatePluginGDBView::slotSendCommand()
{
    QString cmd = m_inputArea->currentText();

    if (cmd.isEmpty()) cmd = m_lastCommand;

    m_inputArea->addToHistory(cmd);
    m_inputArea->setCurrentItem(QString());
    m_focusOnInput = true;
    m_lastCommand = cmd;
    m_debugView->issueCommand(cmd);

    QScrollBar *sb = m_outputArea->verticalScrollBar();
    sb->setValue(sb->maximum());
}

void KatePluginGDBView::insertStackFrame(QString const& level, QString const& info)
{
    if (level.isEmpty() && info.isEmpty()) {
        m_stackTree->resizeColumnToContents(2);
        return;
    }

    if (level == QLatin1String("0"))
    {
        m_stackTree->clear();
    }
    QStringList columns;
    columns << QStringLiteral("  "); // icon place holder
    columns << level;
    int lastSpace = info.lastIndexOf(QLatin1String(" "));
    QString shortInfo = info.mid(lastSpace);
    columns << shortInfo;

    QTreeWidgetItem *item = new QTreeWidgetItem(columns);
    item->setToolTip(2, QStringLiteral("<qt>%1<qt>").arg(info));
    m_stackTree->insertTopLevelItem(level.toInt(), item);
}

void KatePluginGDBView::stackFrameSelected()
{
    m_debugView->issueCommand(QStringLiteral("(Q)f %1").arg(m_stackTree->currentIndex().row()));
}

void KatePluginGDBView::stackFrameChanged(int level)
{
    QTreeWidgetItem *current = m_stackTree->topLevelItem(m_lastExecFrame);
    QTreeWidgetItem *next = m_stackTree->topLevelItem(level);

    if (current) current->setIcon (0, QIcon());
    if (next)    next->setIcon(0, QIcon::fromTheme(QStringLiteral("arrow-right")));
    m_lastExecFrame = level;
}


void KatePluginGDBView::insertThread(int number, bool active)
{
    if (number < 0) {
        m_threadCombo->clear();
        m_activeThread = -1;
        return;
    }
    if (!active) {
        m_threadCombo->addItem(QIcon::fromTheme(QStringLiteral("")).pixmap(10,10),
                               i18n("Thread %1", number), number);
    }
    else {
        m_threadCombo->addItem(QIcon::fromTheme(QStringLiteral("arrow-right")).pixmap(10,10),
                               i18n("Thread %1", number), number);
        m_activeThread = m_threadCombo->count()-1;
    }
    m_threadCombo->setCurrentIndex(m_activeThread);
}

void KatePluginGDBView::threadSelected(int thread)
{
    m_debugView->issueCommand(QStringLiteral("thread %1").
    arg(m_threadCombo->itemData(thread).toInt()));
}

QString KatePluginGDBView::currentWord()
{
    KTextEditor::View *kv = m_mainWin->activeView();
    if (!kv) {
        qDebug() << "no KTextEditor::View" << endl;
        return QString();
    }

    if (!kv->cursorPosition().isValid()) {
        qDebug() << "cursor not valid!" << endl;
        return QString();
    }

    int line = kv->cursorPosition().line();
    int col = kv->cursorPosition().column();

    QString linestr = kv->document()->line(line);

    int startPos = qMax(qMin(col, linestr.length()-1), 0);
    int lindex = linestr.length()-1;
    int endPos = startPos;
    while (startPos >= 0 &&
        (linestr[startPos].isLetterOrNumber() ||
        linestr[startPos] == QLatin1Char('_') ||
        linestr[startPos] == QLatin1Char('~') ||
        ((startPos > 1) && (linestr[startPos] == QLatin1Char('.')) && !linestr[startPos-1].isSpace()) ||
        ((startPos > 2) && (linestr[startPos] == QLatin1Char('>')) && (linestr[startPos-1] == QLatin1Char('-')) && !linestr[startPos-2].isSpace())))
    {
        if (linestr[startPos] == QLatin1Char('>')) {
            startPos--;
        }
        startPos--;
    }
    while (endPos < (int)linestr.length() &&
        (linestr[endPos].isLetterOrNumber() ||
        linestr[endPos] == QLatin1Char('_') ||
        ((endPos < lindex-1) && (linestr[endPos] == QLatin1Char('.')) && !linestr[endPos+1].isSpace()) ||
        ((endPos < lindex-2) && (linestr[endPos] == QLatin1Char('-')) && (linestr[endPos+1] == QLatin1Char('>')) && !linestr[endPos+2].isSpace()) ||
        ((endPos > 1) && (linestr[endPos-1] == QLatin1Char('-')) && (linestr[endPos] == QLatin1Char('>')))
       ))
    {
        if (linestr[endPos] == QLatin1Char('-')) {
            endPos++;
        }
        endPos++;
    }
    if  (startPos == endPos) {
        qDebug() << "no word found!" << endl;
        return QString();
    }

    //qDebug() << linestr.mid(startPos+1, endPos-startPos-1);
    return linestr.mid(startPos+1, endPos-startPos-1);
}

void KatePluginGDBView::slotValue()
{
    QString variable;
    KTextEditor::View* editView = m_mainWin->activeView();
    if (editView && editView->selection() && editView->selectionRange().onSingleLine()) {
        variable = editView->selectionText();
    }

    if (variable.isEmpty()) variable = currentWord();

    if (variable.isEmpty()) return;

    QString cmd = QStringLiteral("print %1").arg(variable);
    m_debugView->issueCommand(cmd);
    m_inputArea->addToHistory(cmd);
    m_inputArea->setCurrentItem(QString());

    m_mainWin->showToolView(m_toolView);
    m_tabWidget->setCurrentWidget(m_gdbPage);

    QScrollBar *sb = m_outputArea->verticalScrollBar();
    sb->setValue(sb->maximum());
}

void KatePluginGDBView::showIO(bool show)
{
    if (show)
    {
        m_tabWidget->addTab(m_ioView, i18n("IO"));
    }
    else
    {
        m_tabWidget->removeTab(m_tabWidget->indexOf(m_ioView));
    }
}

void KatePluginGDBView::addOutputText(QString const& text)
{
    QScrollBar *scrollb = m_outputArea->verticalScrollBar();
    if (!scrollb) return;
    bool atEnd = (scrollb->value() == scrollb->maximum());

    QTextCursor cursor = m_outputArea->textCursor();
    if (!cursor.atEnd()) cursor.movePosition(QTextCursor::End);
    cursor.insertText(text);

    if (atEnd) {
        scrollb->setValue(scrollb->maximum());
    }
}

void KatePluginGDBView::addErrorText(QString const& text)
{
    m_outputArea->setFontItalic(true);
    addOutputText(text);
    m_outputArea->setFontItalic(false);
}

bool KatePluginGDBView::eventFilter(QObject *obj, QEvent *event)
{
    if (event->type() == QEvent::KeyPress) {
        QKeyEvent *ke = static_cast<QKeyEvent*>(event);
        if ((obj == m_toolView) && (ke->key() == Qt::Key_Escape)) {
            m_mainWin->hideToolView(m_toolView);
            event->accept();
            return true;
        }
    }
    return QObject::eventFilter(obj, event);
}

void KatePluginGDBView::handleEsc(QEvent *e)
{
    if (!m_mainWin) return;

    QKeyEvent *k = static_cast<QKeyEvent *>(e);
    if (k->key() == Qt::Key_Escape && k->modifiers() == Qt::NoModifier) {
        if (m_toolView->isVisible()) {
            m_mainWin->hideToolView(m_toolView);
        }
    }
}

#include "plugin_kategdb.moc"

