(function() {
  var escope, expect, harmony;

  expect = require('chai').expect;

  harmony = require('../third_party/esprima');

  escope = require('..');

  describe('export declaration', function() {
    it('should create vairable bindings', function() {
      var ast, globalScope, scope, scopeManager;
      ast = harmony.parse("export var v;", {
        sourceType: 'module'
      });
      scopeManager = escope.analyze(ast, {
        ecmaVersion: 6,
        sourceType: 'module'
      });
      expect(scopeManager.scopes).to.have.length(2);
      globalScope = scopeManager.scopes[0];
      expect(globalScope.type).to.be.equal('global');
      expect(globalScope.variables).to.have.length(0);
      expect(globalScope.references).to.have.length(0);
      scope = scopeManager.scopes[1];
      expect(scope.type).to.be.equal('module');
      expect(scope.variables).to.have.length(1);
      expect(scope.variables[0].name).to.be.equal('v');
      expect(scope.variables[0].defs[0].type).to.be.equal('Variable');
      return expect(scope.references).to.have.length(0);
    });
    it('should create function declaration bindings', function() {
      var ast, globalScope, scope, scopeManager;
      ast = harmony.parse("export default function f(){};", {
        sourceType: 'module'
      });
      scopeManager = escope.analyze(ast, {
        ecmaVersion: 6,
        sourceType: 'module'
      });
      expect(scopeManager.scopes).to.have.length(3);
      globalScope = scopeManager.scopes[0];
      expect(globalScope.type).to.be.equal('global');
      expect(globalScope.variables).to.have.length(0);
      expect(globalScope.references).to.have.length(0);
      scope = scopeManager.scopes[1];
      expect(scope.type).to.be.equal('module');
      expect(scope.variables).to.have.length(1);
      expect(scope.variables[0].name).to.be.equal('f');
      expect(scope.variables[0].defs[0].type).to.be.equal('FunctionName');
      expect(scope.references).to.have.length(0);
      scope = scopeManager.scopes[2];
      expect(scope.type).to.be.equal('function');
      expect(scope.variables).to.have.length(1);
      expect(scope.variables[0].name).to.be.equal('arguments');
      return expect(scope.references).to.have.length(0);
    });
    it('should export function expression', function() {
      var ast, globalScope, scope, scopeManager;
      ast = harmony.parse("export default function(){};", {
        sourceType: 'module'
      });
      scopeManager = escope.analyze(ast, {
        ecmaVersion: 6,
        sourceType: 'module'
      });
      expect(scopeManager.scopes).to.have.length(3);
      globalScope = scopeManager.scopes[0];
      expect(globalScope.type).to.be.equal('global');
      expect(globalScope.variables).to.have.length(0);
      expect(globalScope.references).to.have.length(0);
      scope = scopeManager.scopes[1];
      expect(scope.type).to.be.equal('module');
      expect(scope.variables).to.have.length(0);
      expect(scope.references).to.have.length(0);
      scope = scopeManager.scopes[2];
      expect(scope.type).to.be.equal('function');
      expect(scope.variables).to.have.length(1);
      expect(scope.variables[0].name).to.be.equal('arguments');
      return expect(scope.references).to.have.length(0);
    });
    it('should export literal', function() {
      var ast, globalScope, scope, scopeManager;
      ast = harmony.parse("export default 42;", {
        sourceType: 'module'
      });
      scopeManager = escope.analyze(ast, {
        ecmaVersion: 6,
        sourceType: 'module'
      });
      expect(scopeManager.scopes).to.have.length(2);
      globalScope = scopeManager.scopes[0];
      expect(globalScope.type).to.be.equal('global');
      expect(globalScope.variables).to.have.length(0);
      expect(globalScope.references).to.have.length(0);
      scope = scopeManager.scopes[1];
      expect(scope.type).to.be.equal('module');
      expect(scope.variables).to.have.length(0);
      return expect(scope.references).to.have.length(0);
    });
    it('should refer exported references#1', function() {
      var ast, globalScope, scope, scopeManager;
      ast = harmony.parse("export {x};", {
        sourceType: 'module'
      });
      scopeManager = escope.analyze(ast, {
        ecmaVersion: 6,
        sourceType: 'module'
      });
      expect(scopeManager.scopes).to.have.length(2);
      globalScope = scopeManager.scopes[0];
      expect(globalScope.type).to.be.equal('global');
      expect(globalScope.variables).to.have.length(0);
      expect(globalScope.references).to.have.length(0);
      scope = scopeManager.scopes[1];
      expect(scope.type).to.be.equal('module');
      expect(scope.variables).to.have.length(0);
      expect(scope.references).to.have.length(1);
      return expect(scope.references[0].identifier.name).to.be.equal('x');
    });
    it('should refer exported references#2', function() {
      var ast, globalScope, scope, scopeManager;
      ast = harmony.parse("export {v as x};", {
        sourceType: 'module'
      });
      scopeManager = escope.analyze(ast, {
        ecmaVersion: 6,
        sourceType: 'module'
      });
      expect(scopeManager.scopes).to.have.length(2);
      globalScope = scopeManager.scopes[0];
      expect(globalScope.type).to.be.equal('global');
      expect(globalScope.variables).to.have.length(0);
      expect(globalScope.references).to.have.length(0);
      scope = scopeManager.scopes[1];
      expect(scope.type).to.be.equal('module');
      expect(scope.variables).to.have.length(0);
      expect(scope.references).to.have.length(1);
      return expect(scope.references[0].identifier.name).to.be.equal('v');
    });
    it('should not refer exported references from other source#1', function() {
      var ast, globalScope, scope, scopeManager;
      ast = harmony.parse("export {x} from \"mod\";", {
        sourceType: 'module'
      });
      scopeManager = escope.analyze(ast, {
        ecmaVersion: 6,
        sourceType: 'module'
      });
      expect(scopeManager.scopes).to.have.length(2);
      globalScope = scopeManager.scopes[0];
      expect(globalScope.type).to.be.equal('global');
      expect(globalScope.variables).to.have.length(0);
      expect(globalScope.references).to.have.length(0);
      scope = scopeManager.scopes[1];
      expect(scope.type).to.be.equal('module');
      expect(scope.variables).to.have.length(0);
      return expect(scope.references).to.have.length(0);
    });
    it('should not refer exported references from other source#2', function() {
      var ast, globalScope, scope, scopeManager;
      ast = harmony.parse("export {v as x} from \"mod\";", {
        sourceType: 'module'
      });
      scopeManager = escope.analyze(ast, {
        ecmaVersion: 6,
        sourceType: 'module'
      });
      expect(scopeManager.scopes).to.have.length(2);
      globalScope = scopeManager.scopes[0];
      expect(globalScope.type).to.be.equal('global');
      expect(globalScope.variables).to.have.length(0);
      expect(globalScope.references).to.have.length(0);
      scope = scopeManager.scopes[1];
      expect(scope.type).to.be.equal('module');
      expect(scope.variables).to.have.length(0);
      return expect(scope.references).to.have.length(0);
    });
    return it('should not refer exported references from other source#3', function() {
      var ast, globalScope, scope, scopeManager;
      ast = harmony.parse("export * from \"mod\";", {
        sourceType: 'module'
      });
      scopeManager = escope.analyze(ast, {
        ecmaVersion: 6,
        sourceType: 'module'
      });
      expect(scopeManager.scopes).to.have.length(2);
      globalScope = scopeManager.scopes[0];
      expect(globalScope.type).to.be.equal('global');
      expect(globalScope.variables).to.have.length(0);
      expect(globalScope.references).to.have.length(0);
      scope = scopeManager.scopes[1];
      expect(scope.type).to.be.equal('module');
      expect(scope.variables).to.have.length(0);
      return expect(scope.references).to.have.length(0);
    });
  });

}).call(this);

//# sourceMappingURL=data:application/json;base64,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